﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/ecr-public/ECRPublic_EXPORTS.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/core/utils/Array.h>
#include <utility>

namespace Aws
{
namespace Utils
{
namespace Json
{
  class JsonValue;
  class JsonView;
} // namespace Json
} // namespace Utils
namespace ECRPublic
{
namespace Model
{

  /**
   * <p>An object that contains the catalog data for a repository. This data is
   * publicly visible in the Amazon ECR Public Gallery.</p><p><h3>See Also:</h3>   <a
   * href="http://docs.aws.amazon.com/goto/WebAPI/ecr-public-2020-10-30/RepositoryCatalogDataInput">AWS
   * API Reference</a></p>
   */
  class RepositoryCatalogDataInput
  {
  public:
    AWS_ECRPUBLIC_API RepositoryCatalogDataInput();
    AWS_ECRPUBLIC_API RepositoryCatalogDataInput(Aws::Utils::Json::JsonView jsonValue);
    AWS_ECRPUBLIC_API RepositoryCatalogDataInput& operator=(Aws::Utils::Json::JsonView jsonValue);
    AWS_ECRPUBLIC_API Aws::Utils::Json::JsonValue Jsonize() const;


    ///@{
    /**
     * <p>A short description of the contents of the repository. This text appears in
     * both the image details and also when searching for repositories on the Amazon
     * ECR Public Gallery.</p>
     */
    inline const Aws::String& GetDescription() const{ return m_description; }
    inline bool DescriptionHasBeenSet() const { return m_descriptionHasBeenSet; }
    inline void SetDescription(const Aws::String& value) { m_descriptionHasBeenSet = true; m_description = value; }
    inline void SetDescription(Aws::String&& value) { m_descriptionHasBeenSet = true; m_description = std::move(value); }
    inline void SetDescription(const char* value) { m_descriptionHasBeenSet = true; m_description.assign(value); }
    inline RepositoryCatalogDataInput& WithDescription(const Aws::String& value) { SetDescription(value); return *this;}
    inline RepositoryCatalogDataInput& WithDescription(Aws::String&& value) { SetDescription(std::move(value)); return *this;}
    inline RepositoryCatalogDataInput& WithDescription(const char* value) { SetDescription(value); return *this;}
    ///@}

    ///@{
    /**
     * <p>The system architecture that the images in the repository are compatible
     * with. On the Amazon ECR Public Gallery, the following supported architectures
     * appear as badges on the repository and are used as search filters.</p> 
     * <p>If an unsupported tag is added to your repository catalog data, it's
     * associated with the repository and can be retrieved using the API but isn't
     * discoverable in the Amazon ECR Public Gallery.</p>  <ul> <li> <p>
     * <code>ARM</code> </p> </li> <li> <p> <code>ARM 64</code> </p> </li> <li> <p>
     * <code>x86</code> </p> </li> <li> <p> <code>x86-64</code> </p> </li> </ul>
     */
    inline const Aws::Vector<Aws::String>& GetArchitectures() const{ return m_architectures; }
    inline bool ArchitecturesHasBeenSet() const { return m_architecturesHasBeenSet; }
    inline void SetArchitectures(const Aws::Vector<Aws::String>& value) { m_architecturesHasBeenSet = true; m_architectures = value; }
    inline void SetArchitectures(Aws::Vector<Aws::String>&& value) { m_architecturesHasBeenSet = true; m_architectures = std::move(value); }
    inline RepositoryCatalogDataInput& WithArchitectures(const Aws::Vector<Aws::String>& value) { SetArchitectures(value); return *this;}
    inline RepositoryCatalogDataInput& WithArchitectures(Aws::Vector<Aws::String>&& value) { SetArchitectures(std::move(value)); return *this;}
    inline RepositoryCatalogDataInput& AddArchitectures(const Aws::String& value) { m_architecturesHasBeenSet = true; m_architectures.push_back(value); return *this; }
    inline RepositoryCatalogDataInput& AddArchitectures(Aws::String&& value) { m_architecturesHasBeenSet = true; m_architectures.push_back(std::move(value)); return *this; }
    inline RepositoryCatalogDataInput& AddArchitectures(const char* value) { m_architecturesHasBeenSet = true; m_architectures.push_back(value); return *this; }
    ///@}

    ///@{
    /**
     * <p>The operating systems that the images in the repository are compatible with.
     * On the Amazon ECR Public Gallery, the following supported operating systems
     * appear as badges on the repository and are used as search filters.</p> 
     * <p>If an unsupported tag is added to your repository catalog data, it's
     * associated with the repository and can be retrieved using the API but isn't
     * discoverable in the Amazon ECR Public Gallery.</p>  <ul> <li> <p>
     * <code>Linux</code> </p> </li> <li> <p> <code>Windows</code> </p> </li> </ul>
     */
    inline const Aws::Vector<Aws::String>& GetOperatingSystems() const{ return m_operatingSystems; }
    inline bool OperatingSystemsHasBeenSet() const { return m_operatingSystemsHasBeenSet; }
    inline void SetOperatingSystems(const Aws::Vector<Aws::String>& value) { m_operatingSystemsHasBeenSet = true; m_operatingSystems = value; }
    inline void SetOperatingSystems(Aws::Vector<Aws::String>&& value) { m_operatingSystemsHasBeenSet = true; m_operatingSystems = std::move(value); }
    inline RepositoryCatalogDataInput& WithOperatingSystems(const Aws::Vector<Aws::String>& value) { SetOperatingSystems(value); return *this;}
    inline RepositoryCatalogDataInput& WithOperatingSystems(Aws::Vector<Aws::String>&& value) { SetOperatingSystems(std::move(value)); return *this;}
    inline RepositoryCatalogDataInput& AddOperatingSystems(const Aws::String& value) { m_operatingSystemsHasBeenSet = true; m_operatingSystems.push_back(value); return *this; }
    inline RepositoryCatalogDataInput& AddOperatingSystems(Aws::String&& value) { m_operatingSystemsHasBeenSet = true; m_operatingSystems.push_back(std::move(value)); return *this; }
    inline RepositoryCatalogDataInput& AddOperatingSystems(const char* value) { m_operatingSystemsHasBeenSet = true; m_operatingSystems.push_back(value); return *this; }
    ///@}

    ///@{
    /**
     * <p>The base64-encoded repository logo payload.</p>  <p>The repository logo
     * is only publicly visible in the Amazon ECR Public Gallery for verified
     * accounts.</p> 
     */
    inline const Aws::Utils::ByteBuffer& GetLogoImageBlob() const{ return m_logoImageBlob; }
    inline bool LogoImageBlobHasBeenSet() const { return m_logoImageBlobHasBeenSet; }
    inline void SetLogoImageBlob(const Aws::Utils::ByteBuffer& value) { m_logoImageBlobHasBeenSet = true; m_logoImageBlob = value; }
    inline void SetLogoImageBlob(Aws::Utils::ByteBuffer&& value) { m_logoImageBlobHasBeenSet = true; m_logoImageBlob = std::move(value); }
    inline RepositoryCatalogDataInput& WithLogoImageBlob(const Aws::Utils::ByteBuffer& value) { SetLogoImageBlob(value); return *this;}
    inline RepositoryCatalogDataInput& WithLogoImageBlob(Aws::Utils::ByteBuffer&& value) { SetLogoImageBlob(std::move(value)); return *this;}
    ///@}

    ///@{
    /**
     * <p>A detailed description of the contents of the repository. It's publicly
     * visible in the Amazon ECR Public Gallery. The text must be in markdown
     * format.</p>
     */
    inline const Aws::String& GetAboutText() const{ return m_aboutText; }
    inline bool AboutTextHasBeenSet() const { return m_aboutTextHasBeenSet; }
    inline void SetAboutText(const Aws::String& value) { m_aboutTextHasBeenSet = true; m_aboutText = value; }
    inline void SetAboutText(Aws::String&& value) { m_aboutTextHasBeenSet = true; m_aboutText = std::move(value); }
    inline void SetAboutText(const char* value) { m_aboutTextHasBeenSet = true; m_aboutText.assign(value); }
    inline RepositoryCatalogDataInput& WithAboutText(const Aws::String& value) { SetAboutText(value); return *this;}
    inline RepositoryCatalogDataInput& WithAboutText(Aws::String&& value) { SetAboutText(std::move(value)); return *this;}
    inline RepositoryCatalogDataInput& WithAboutText(const char* value) { SetAboutText(value); return *this;}
    ///@}

    ///@{
    /**
     * <p>Detailed information about how to use the contents of the repository. It's
     * publicly visible in the Amazon ECR Public Gallery. The usage text provides
     * context, support information, and additional usage details for users of the
     * repository. The text must be in markdown format.</p>
     */
    inline const Aws::String& GetUsageText() const{ return m_usageText; }
    inline bool UsageTextHasBeenSet() const { return m_usageTextHasBeenSet; }
    inline void SetUsageText(const Aws::String& value) { m_usageTextHasBeenSet = true; m_usageText = value; }
    inline void SetUsageText(Aws::String&& value) { m_usageTextHasBeenSet = true; m_usageText = std::move(value); }
    inline void SetUsageText(const char* value) { m_usageTextHasBeenSet = true; m_usageText.assign(value); }
    inline RepositoryCatalogDataInput& WithUsageText(const Aws::String& value) { SetUsageText(value); return *this;}
    inline RepositoryCatalogDataInput& WithUsageText(Aws::String&& value) { SetUsageText(std::move(value)); return *this;}
    inline RepositoryCatalogDataInput& WithUsageText(const char* value) { SetUsageText(value); return *this;}
    ///@}
  private:

    Aws::String m_description;
    bool m_descriptionHasBeenSet = false;

    Aws::Vector<Aws::String> m_architectures;
    bool m_architecturesHasBeenSet = false;

    Aws::Vector<Aws::String> m_operatingSystems;
    bool m_operatingSystemsHasBeenSet = false;

    Aws::Utils::ByteBuffer m_logoImageBlob;
    bool m_logoImageBlobHasBeenSet = false;

    Aws::String m_aboutText;
    bool m_aboutTextHasBeenSet = false;

    Aws::String m_usageText;
    bool m_usageTextHasBeenSet = false;
  };

} // namespace Model
} // namespace ECRPublic
} // namespace Aws
