﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/bedrock/Bedrock_EXPORTS.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <utility>

namespace Aws
{
namespace Utils
{
namespace Json
{
  class JsonValue;
  class JsonView;
} // namespace Json
} // namespace Utils
namespace Bedrock
{
namespace Model
{

  /**
   * <p>The configuration details for text generation using a language model via the
   * <code>RetrieveAndGenerate</code> function.</p><p><h3>See Also:</h3>   <a
   * href="http://docs.aws.amazon.com/goto/WebAPI/bedrock-2023-04-20/TextInferenceConfig">AWS
   * API Reference</a></p>
   */
  class TextInferenceConfig
  {
  public:
    AWS_BEDROCK_API TextInferenceConfig() = default;
    AWS_BEDROCK_API TextInferenceConfig(Aws::Utils::Json::JsonView jsonValue);
    AWS_BEDROCK_API TextInferenceConfig& operator=(Aws::Utils::Json::JsonView jsonValue);
    AWS_BEDROCK_API Aws::Utils::Json::JsonValue Jsonize() const;


    ///@{
    /**
     * <p>Controls the random-ness of text generated by the language model, influencing
     * how much the model sticks to the most predictable next words versus exploring
     * more surprising options. A lower temperature value (e.g. 0.2 or 0.3) makes model
     * outputs more deterministic or predictable, while a higher temperature (e.g. 0.8
     * or 0.9) makes the outputs more creative or unpredictable.</p>
     */
    inline double GetTemperature() const { return m_temperature; }
    inline bool TemperatureHasBeenSet() const { return m_temperatureHasBeenSet; }
    inline void SetTemperature(double value) { m_temperatureHasBeenSet = true; m_temperature = value; }
    inline TextInferenceConfig& WithTemperature(double value) { SetTemperature(value); return *this;}
    ///@}

    ///@{
    /**
     * <p>A probability distribution threshold which controls what the model considers
     * for the set of possible next tokens. The model will only consider the top p% of
     * the probability distribution when generating the next token.</p>
     */
    inline double GetTopP() const { return m_topP; }
    inline bool TopPHasBeenSet() const { return m_topPHasBeenSet; }
    inline void SetTopP(double value) { m_topPHasBeenSet = true; m_topP = value; }
    inline TextInferenceConfig& WithTopP(double value) { SetTopP(value); return *this;}
    ///@}

    ///@{
    /**
     * <p>The maximum number of tokens to generate in the output text. Do not use the
     * minimum of 0 or the maximum of 65536. The limit values described here are
     * arbitrary values, for actual values consult the limits defined by your specific
     * model.</p>
     */
    inline int GetMaxTokens() const { return m_maxTokens; }
    inline bool MaxTokensHasBeenSet() const { return m_maxTokensHasBeenSet; }
    inline void SetMaxTokens(int value) { m_maxTokensHasBeenSet = true; m_maxTokens = value; }
    inline TextInferenceConfig& WithMaxTokens(int value) { SetMaxTokens(value); return *this;}
    ///@}

    ///@{
    /**
     * <p>A list of sequences of characters that, if generated, will cause the model to
     * stop generating further tokens. Do not use a minimum length of 1 or a maximum
     * length of 1000. The limit values described here are arbitrary values, for actual
     * values consult the limits defined by your specific model.</p>
     */
    inline const Aws::Vector<Aws::String>& GetStopSequences() const { return m_stopSequences; }
    inline bool StopSequencesHasBeenSet() const { return m_stopSequencesHasBeenSet; }
    template<typename StopSequencesT = Aws::Vector<Aws::String>>
    void SetStopSequences(StopSequencesT&& value) { m_stopSequencesHasBeenSet = true; m_stopSequences = std::forward<StopSequencesT>(value); }
    template<typename StopSequencesT = Aws::Vector<Aws::String>>
    TextInferenceConfig& WithStopSequences(StopSequencesT&& value) { SetStopSequences(std::forward<StopSequencesT>(value)); return *this;}
    template<typename StopSequencesT = Aws::String>
    TextInferenceConfig& AddStopSequences(StopSequencesT&& value) { m_stopSequencesHasBeenSet = true; m_stopSequences.emplace_back(std::forward<StopSequencesT>(value)); return *this; }
    ///@}
  private:

    double m_temperature{0.0};
    bool m_temperatureHasBeenSet = false;

    double m_topP{0.0};
    bool m_topPHasBeenSet = false;

    int m_maxTokens{0};
    bool m_maxTokensHasBeenSet = false;

    Aws::Vector<Aws::String> m_stopSequences;
    bool m_stopSequencesHasBeenSet = false;
  };

} // namespace Model
} // namespace Bedrock
} // namespace Aws
