﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/customer-profiles/CustomerProfilesRequest.h>
#include <aws/customer-profiles/CustomerProfiles_EXPORTS.h>
#include <aws/customer-profiles/model/IntegrationConfig.h>
#include <aws/customer-profiles/model/WorkflowType.h>

#include <utility>

namespace Aws {
namespace CustomerProfiles {
namespace Model {

/**
 */
class CreateIntegrationWorkflowRequest : public CustomerProfilesRequest {
 public:
  AWS_CUSTOMERPROFILES_API CreateIntegrationWorkflowRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateIntegrationWorkflow"; }

  AWS_CUSTOMERPROFILES_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The unique name of the domain.</p>
   */
  inline const Aws::String& GetDomainName() const { return m_domainName; }
  inline bool DomainNameHasBeenSet() const { return m_domainNameHasBeenSet; }
  template <typename DomainNameT = Aws::String>
  void SetDomainName(DomainNameT&& value) {
    m_domainNameHasBeenSet = true;
    m_domainName = std::forward<DomainNameT>(value);
  }
  template <typename DomainNameT = Aws::String>
  CreateIntegrationWorkflowRequest& WithDomainName(DomainNameT&& value) {
    SetDomainName(std::forward<DomainNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The type of workflow. The only supported value is APPFLOW_INTEGRATION.</p>
   */
  inline WorkflowType GetWorkflowType() const { return m_workflowType; }
  inline bool WorkflowTypeHasBeenSet() const { return m_workflowTypeHasBeenSet; }
  inline void SetWorkflowType(WorkflowType value) {
    m_workflowTypeHasBeenSet = true;
    m_workflowType = value;
  }
  inline CreateIntegrationWorkflowRequest& WithWorkflowType(WorkflowType value) {
    SetWorkflowType(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Configuration data for integration workflow.</p>
   */
  inline const IntegrationConfig& GetIntegrationConfig() const { return m_integrationConfig; }
  inline bool IntegrationConfigHasBeenSet() const { return m_integrationConfigHasBeenSet; }
  template <typename IntegrationConfigT = IntegrationConfig>
  void SetIntegrationConfig(IntegrationConfigT&& value) {
    m_integrationConfigHasBeenSet = true;
    m_integrationConfig = std::forward<IntegrationConfigT>(value);
  }
  template <typename IntegrationConfigT = IntegrationConfig>
  CreateIntegrationWorkflowRequest& WithIntegrationConfig(IntegrationConfigT&& value) {
    SetIntegrationConfig(std::forward<IntegrationConfigT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the profile object type.</p>
   */
  inline const Aws::String& GetObjectTypeName() const { return m_objectTypeName; }
  inline bool ObjectTypeNameHasBeenSet() const { return m_objectTypeNameHasBeenSet; }
  template <typename ObjectTypeNameT = Aws::String>
  void SetObjectTypeName(ObjectTypeNameT&& value) {
    m_objectTypeNameHasBeenSet = true;
    m_objectTypeName = std::forward<ObjectTypeNameT>(value);
  }
  template <typename ObjectTypeNameT = Aws::String>
  CreateIntegrationWorkflowRequest& WithObjectTypeName(ObjectTypeNameT&& value) {
    SetObjectTypeName(std::forward<ObjectTypeNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the IAM role. Customer Profiles assumes
   * this role to create resources on your behalf as part of workflow execution.</p>
   */
  inline const Aws::String& GetRoleArn() const { return m_roleArn; }
  inline bool RoleArnHasBeenSet() const { return m_roleArnHasBeenSet; }
  template <typename RoleArnT = Aws::String>
  void SetRoleArn(RoleArnT&& value) {
    m_roleArnHasBeenSet = true;
    m_roleArn = std::forward<RoleArnT>(value);
  }
  template <typename RoleArnT = Aws::String>
  CreateIntegrationWorkflowRequest& WithRoleArn(RoleArnT&& value) {
    SetRoleArn(std::forward<RoleArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The tags used to organize, track, or control access for this resource.</p>
   */
  inline const Aws::Map<Aws::String, Aws::String>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  CreateIntegrationWorkflowRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsKeyT = Aws::String, typename TagsValueT = Aws::String>
  CreateIntegrationWorkflowRequest& AddTags(TagsKeyT&& key, TagsValueT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace(std::forward<TagsKeyT>(key), std::forward<TagsValueT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_domainName;
  bool m_domainNameHasBeenSet = false;

  WorkflowType m_workflowType{WorkflowType::NOT_SET};
  bool m_workflowTypeHasBeenSet = false;

  IntegrationConfig m_integrationConfig;
  bool m_integrationConfigHasBeenSet = false;

  Aws::String m_objectTypeName;
  bool m_objectTypeNameHasBeenSet = false;

  Aws::String m_roleArn;
  bool m_roleArnHasBeenSet = false;

  Aws::Map<Aws::String, Aws::String> m_tags;
  bool m_tagsHasBeenSet = false;
};

}  // namespace Model
}  // namespace CustomerProfiles
}  // namespace Aws
