﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/globalaccelerator/GlobalAccelerator_EXPORTS.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace GlobalAccelerator {
namespace Model {

/**
 * <p>A resource is one of the following: the ARN for an Amazon Web Services
 * resource that is supported by Global Accelerator to be added as an endpoint, or
 * a CIDR range that specifies a bring your own IP (BYOIP) address
 * pool.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/globalaccelerator-2018-08-08/Resource">AWS
 * API Reference</a></p>
 */
class Resource {
 public:
  AWS_GLOBALACCELERATOR_API Resource() = default;
  AWS_GLOBALACCELERATOR_API Resource(Aws::Utils::Json::JsonView jsonValue);
  AWS_GLOBALACCELERATOR_API Resource& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_GLOBALACCELERATOR_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>The endpoint ID for the endpoint that is specified as a Amazon Web Services
   * resource. </p> <p>An endpoint ID for the cross-account feature is the ARN of an
   * Amazon Web Services resource, such as a Network Load Balancer, that Global
   * Accelerator supports as an endpoint for an accelerator.</p>
   */
  inline const Aws::String& GetEndpointId() const { return m_endpointId; }
  inline bool EndpointIdHasBeenSet() const { return m_endpointIdHasBeenSet; }
  template <typename EndpointIdT = Aws::String>
  void SetEndpointId(EndpointIdT&& value) {
    m_endpointIdHasBeenSet = true;
    m_endpointId = std::forward<EndpointIdT>(value);
  }
  template <typename EndpointIdT = Aws::String>
  Resource& WithEndpointId(EndpointIdT&& value) {
    SetEndpointId(std::forward<EndpointIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>An IP address range, in CIDR format, that is specified as resource. The
   * address must be provisioned and advertised in Global Accelerator by following
   * the bring your own IP address (BYOIP) process for Global Accelerator</p> <p> For
   * more information, see <a
   * href="https://docs.aws.amazon.com/global-accelerator/latest/dg/using-byoip.html">Bring
   * your own IP addresses (BYOIP)</a> in the Global Accelerator Developer Guide.</p>
   */
  inline const Aws::String& GetCidr() const { return m_cidr; }
  inline bool CidrHasBeenSet() const { return m_cidrHasBeenSet; }
  template <typename CidrT = Aws::String>
  void SetCidr(CidrT&& value) {
    m_cidrHasBeenSet = true;
    m_cidr = std::forward<CidrT>(value);
  }
  template <typename CidrT = Aws::String>
  Resource& WithCidr(CidrT&& value) {
    SetCidr(std::forward<CidrT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Web Services Region where a shared endpoint resource is
   * located.</p>
   */
  inline const Aws::String& GetRegion() const { return m_region; }
  inline bool RegionHasBeenSet() const { return m_regionHasBeenSet; }
  template <typename RegionT = Aws::String>
  void SetRegion(RegionT&& value) {
    m_regionHasBeenSet = true;
    m_region = std::forward<RegionT>(value);
  }
  template <typename RegionT = Aws::String>
  Resource& WithRegion(RegionT&& value) {
    SetRegion(std::forward<RegionT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_endpointId;
  bool m_endpointIdHasBeenSet = false;

  Aws::String m_cidr;
  bool m_cidrHasBeenSet = false;

  Aws::String m_region;
  bool m_regionHasBeenSet = false;
};

}  // namespace Model
}  // namespace GlobalAccelerator
}  // namespace Aws
