﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/glue/GlueRequest.h>
#include <aws/glue/Glue_EXPORTS.h>
#include <aws/glue/model/DataCatalogEncryptionSettings.h>

#include <utility>

namespace Aws {
namespace Glue {
namespace Model {

/**
 */
class PutDataCatalogEncryptionSettingsRequest : public GlueRequest {
 public:
  AWS_GLUE_API PutDataCatalogEncryptionSettingsRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "PutDataCatalogEncryptionSettings"; }

  AWS_GLUE_API Aws::String SerializePayload() const override;

  AWS_GLUE_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The ID of the Data Catalog to set the security configuration for. If none is
   * provided, the Amazon Web Services account ID is used by default.</p>
   */
  inline const Aws::String& GetCatalogId() const { return m_catalogId; }
  inline bool CatalogIdHasBeenSet() const { return m_catalogIdHasBeenSet; }
  template <typename CatalogIdT = Aws::String>
  void SetCatalogId(CatalogIdT&& value) {
    m_catalogIdHasBeenSet = true;
    m_catalogId = std::forward<CatalogIdT>(value);
  }
  template <typename CatalogIdT = Aws::String>
  PutDataCatalogEncryptionSettingsRequest& WithCatalogId(CatalogIdT&& value) {
    SetCatalogId(std::forward<CatalogIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The security configuration to set.</p>
   */
  inline const DataCatalogEncryptionSettings& GetDataCatalogEncryptionSettings() const { return m_dataCatalogEncryptionSettings; }
  inline bool DataCatalogEncryptionSettingsHasBeenSet() const { return m_dataCatalogEncryptionSettingsHasBeenSet; }
  template <typename DataCatalogEncryptionSettingsT = DataCatalogEncryptionSettings>
  void SetDataCatalogEncryptionSettings(DataCatalogEncryptionSettingsT&& value) {
    m_dataCatalogEncryptionSettingsHasBeenSet = true;
    m_dataCatalogEncryptionSettings = std::forward<DataCatalogEncryptionSettingsT>(value);
  }
  template <typename DataCatalogEncryptionSettingsT = DataCatalogEncryptionSettings>
  PutDataCatalogEncryptionSettingsRequest& WithDataCatalogEncryptionSettings(DataCatalogEncryptionSettingsT&& value) {
    SetDataCatalogEncryptionSettings(std::forward<DataCatalogEncryptionSettingsT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_catalogId;
  bool m_catalogIdHasBeenSet = false;

  DataCatalogEncryptionSettings m_dataCatalogEncryptionSettings;
  bool m_dataCatalogEncryptionSettingsHasBeenSet = false;
};

}  // namespace Model
}  // namespace Glue
}  // namespace Aws
