﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/inspector/InspectorRequest.h>
#include <aws/inspector/Inspector_EXPORTS.h>
#include <aws/inspector/model/InspectorEvent.h>

#include <utility>

namespace Aws {
namespace Inspector {
namespace Model {

/**
 */
class UnsubscribeFromEventRequest : public InspectorRequest {
 public:
  AWS_INSPECTOR_API UnsubscribeFromEventRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "UnsubscribeFromEvent"; }

  AWS_INSPECTOR_API Aws::String SerializePayload() const override;

  AWS_INSPECTOR_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The ARN of the assessment template that is used during the event for which
   * you want to stop receiving SNS notifications.</p>
   */
  inline const Aws::String& GetResourceArn() const { return m_resourceArn; }
  inline bool ResourceArnHasBeenSet() const { return m_resourceArnHasBeenSet; }
  template <typename ResourceArnT = Aws::String>
  void SetResourceArn(ResourceArnT&& value) {
    m_resourceArnHasBeenSet = true;
    m_resourceArn = std::forward<ResourceArnT>(value);
  }
  template <typename ResourceArnT = Aws::String>
  UnsubscribeFromEventRequest& WithResourceArn(ResourceArnT&& value) {
    SetResourceArn(std::forward<ResourceArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The event for which you want to stop receiving SNS notifications.</p>
   */
  inline InspectorEvent GetEvent() const { return m_event; }
  inline bool EventHasBeenSet() const { return m_eventHasBeenSet; }
  inline void SetEvent(InspectorEvent value) {
    m_eventHasBeenSet = true;
    m_event = value;
  }
  inline UnsubscribeFromEventRequest& WithEvent(InspectorEvent value) {
    SetEvent(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ARN of the SNS topic to which SNS notifications are sent.</p>
   */
  inline const Aws::String& GetTopicArn() const { return m_topicArn; }
  inline bool TopicArnHasBeenSet() const { return m_topicArnHasBeenSet; }
  template <typename TopicArnT = Aws::String>
  void SetTopicArn(TopicArnT&& value) {
    m_topicArnHasBeenSet = true;
    m_topicArn = std::forward<TopicArnT>(value);
  }
  template <typename TopicArnT = Aws::String>
  UnsubscribeFromEventRequest& WithTopicArn(TopicArnT&& value) {
    SetTopicArn(std::forward<TopicArnT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_resourceArn;
  bool m_resourceArnHasBeenSet = false;

  InspectorEvent m_event{InspectorEvent::NOT_SET};
  bool m_eventHasBeenSet = false;

  Aws::String m_topicArn;
  bool m_topicArnHasBeenSet = false;
};

}  // namespace Model
}  // namespace Inspector
}  // namespace Aws
