﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/iot/IoTRequest.h>
#include <aws/iot/IoT_EXPORTS.h>
#include <aws/iot/model/Tag.h>

#include <utility>

namespace Aws {
namespace IoT {
namespace Model {

/**
 */
class CreateRoleAliasRequest : public IoTRequest {
 public:
  AWS_IOT_API CreateRoleAliasRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateRoleAlias"; }

  AWS_IOT_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The role alias that points to a role ARN. This allows you to change the role
   * without having to update the device.</p>
   */
  inline const Aws::String& GetRoleAlias() const { return m_roleAlias; }
  inline bool RoleAliasHasBeenSet() const { return m_roleAliasHasBeenSet; }
  template <typename RoleAliasT = Aws::String>
  void SetRoleAlias(RoleAliasT&& value) {
    m_roleAliasHasBeenSet = true;
    m_roleAlias = std::forward<RoleAliasT>(value);
  }
  template <typename RoleAliasT = Aws::String>
  CreateRoleAliasRequest& WithRoleAlias(RoleAliasT&& value) {
    SetRoleAlias(std::forward<RoleAliasT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The role ARN.</p>
   */
  inline const Aws::String& GetRoleArn() const { return m_roleArn; }
  inline bool RoleArnHasBeenSet() const { return m_roleArnHasBeenSet; }
  template <typename RoleArnT = Aws::String>
  void SetRoleArn(RoleArnT&& value) {
    m_roleArnHasBeenSet = true;
    m_roleArn = std::forward<RoleArnT>(value);
  }
  template <typename RoleArnT = Aws::String>
  CreateRoleAliasRequest& WithRoleArn(RoleArnT&& value) {
    SetRoleArn(std::forward<RoleArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>How long (in seconds) the credentials will be valid. The default value is
   * 3,600 seconds.</p> <p>This value must be less than or equal to the maximum
   * session duration of the IAM role that the role alias references.</p>
   */
  inline int GetCredentialDurationSeconds() const { return m_credentialDurationSeconds; }
  inline bool CredentialDurationSecondsHasBeenSet() const { return m_credentialDurationSecondsHasBeenSet; }
  inline void SetCredentialDurationSeconds(int value) {
    m_credentialDurationSecondsHasBeenSet = true;
    m_credentialDurationSeconds = value;
  }
  inline CreateRoleAliasRequest& WithCredentialDurationSeconds(int value) {
    SetCredentialDurationSeconds(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Metadata which can be used to manage the role alias.</p>  <p>For URI
   * Request parameters use format: ...key1=value1&amp;key2=value2...</p> <p>For the
   * CLI command-line parameter use format: &amp;&amp;tags
   * "key1=value1&amp;key2=value2..."</p> <p>For the cli-input-json file use format:
   * "tags": "key1=value1&amp;key2=value2..."</p>
   */
  inline const Aws::Vector<Tag>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Vector<Tag>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Vector<Tag>>
  CreateRoleAliasRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsT = Tag>
  CreateRoleAliasRequest& AddTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace_back(std::forward<TagsT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_roleAlias;
  bool m_roleAliasHasBeenSet = false;

  Aws::String m_roleArn;
  bool m_roleArnHasBeenSet = false;

  int m_credentialDurationSeconds{0};
  bool m_credentialDurationSecondsHasBeenSet = false;

  Aws::Vector<Tag> m_tags;
  bool m_tagsHasBeenSet = false;
};

}  // namespace Model
}  // namespace IoT
}  // namespace Aws
