﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/mediaconnect/MediaConnectRequest.h>
#include <aws/mediaconnect/MediaConnect_EXPORTS.h>
#include <aws/mediaconnect/model/GrantEntitlementRequest.h>

#include <utility>

namespace Aws {
namespace MediaConnect {
namespace Model {

/**
 */
class GrantFlowEntitlementsRequest : public MediaConnectRequest {
 public:
  AWS_MEDIACONNECT_API GrantFlowEntitlementsRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "GrantFlowEntitlements"; }

  AWS_MEDIACONNECT_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p> The list of entitlements that you want to grant.</p>
   */
  inline const Aws::Vector<GrantEntitlementRequest>& GetEntitlements() const { return m_entitlements; }
  inline bool EntitlementsHasBeenSet() const { return m_entitlementsHasBeenSet; }
  template <typename EntitlementsT = Aws::Vector<GrantEntitlementRequest>>
  void SetEntitlements(EntitlementsT&& value) {
    m_entitlementsHasBeenSet = true;
    m_entitlements = std::forward<EntitlementsT>(value);
  }
  template <typename EntitlementsT = Aws::Vector<GrantEntitlementRequest>>
  GrantFlowEntitlementsRequest& WithEntitlements(EntitlementsT&& value) {
    SetEntitlements(std::forward<EntitlementsT>(value));
    return *this;
  }
  template <typename EntitlementsT = GrantEntitlementRequest>
  GrantFlowEntitlementsRequest& AddEntitlements(EntitlementsT&& value) {
    m_entitlementsHasBeenSet = true;
    m_entitlements.emplace_back(std::forward<EntitlementsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> The Amazon Resource Name (ARN) of the flow that you want to grant
   * entitlements on.</p>
   */
  inline const Aws::String& GetFlowArn() const { return m_flowArn; }
  inline bool FlowArnHasBeenSet() const { return m_flowArnHasBeenSet; }
  template <typename FlowArnT = Aws::String>
  void SetFlowArn(FlowArnT&& value) {
    m_flowArnHasBeenSet = true;
    m_flowArn = std::forward<FlowArnT>(value);
  }
  template <typename FlowArnT = Aws::String>
  GrantFlowEntitlementsRequest& WithFlowArn(FlowArnT&& value) {
    SetFlowArn(std::forward<FlowArnT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::Vector<GrantEntitlementRequest> m_entitlements;
  bool m_entitlementsHasBeenSet = false;

  Aws::String m_flowArn;
  bool m_flowArnHasBeenSet = false;
};

}  // namespace Model
}  // namespace MediaConnect
}  // namespace Aws
