﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/DateTime.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/pinpoint-sms-voice-v2/PinpointSMSVoiceV2_EXPORTS.h>
#include <aws/pinpoint-sms-voice-v2/model/MessageType.h>
#include <aws/pinpoint-sms-voice-v2/model/PoolStatus.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace PinpointSMSVoiceV2 {
namespace Model {

/**
 * <p>The information for a pool in an Amazon Web Services account.</p><p><h3>See
 * Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/pinpoint-sms-voice-v2-2022-03-31/PoolInformation">AWS
 * API Reference</a></p>
 */
class PoolInformation {
 public:
  AWS_PINPOINTSMSVOICEV2_API PoolInformation() = default;
  AWS_PINPOINTSMSVOICEV2_API PoolInformation(Aws::Utils::Json::JsonView jsonValue);
  AWS_PINPOINTSMSVOICEV2_API PoolInformation& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_PINPOINTSMSVOICEV2_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) for the pool.</p>
   */
  inline const Aws::String& GetPoolArn() const { return m_poolArn; }
  inline bool PoolArnHasBeenSet() const { return m_poolArnHasBeenSet; }
  template <typename PoolArnT = Aws::String>
  void SetPoolArn(PoolArnT&& value) {
    m_poolArnHasBeenSet = true;
    m_poolArn = std::forward<PoolArnT>(value);
  }
  template <typename PoolArnT = Aws::String>
  PoolInformation& WithPoolArn(PoolArnT&& value) {
    SetPoolArn(std::forward<PoolArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The unique identifier for the pool.</p>
   */
  inline const Aws::String& GetPoolId() const { return m_poolId; }
  inline bool PoolIdHasBeenSet() const { return m_poolIdHasBeenSet; }
  template <typename PoolIdT = Aws::String>
  void SetPoolId(PoolIdT&& value) {
    m_poolIdHasBeenSet = true;
    m_poolId = std::forward<PoolIdT>(value);
  }
  template <typename PoolIdT = Aws::String>
  PoolInformation& WithPoolId(PoolIdT&& value) {
    SetPoolId(std::forward<PoolIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The current status of the pool.</p>
   */
  inline PoolStatus GetStatus() const { return m_status; }
  inline bool StatusHasBeenSet() const { return m_statusHasBeenSet; }
  inline void SetStatus(PoolStatus value) {
    m_statusHasBeenSet = true;
    m_status = value;
  }
  inline PoolInformation& WithStatus(PoolStatus value) {
    SetStatus(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The type of message. Valid values are TRANSACTIONAL for messages that are
   * critical or time-sensitive and PROMOTIONAL for messages that aren't critical or
   * time-sensitive.</p>
   */
  inline MessageType GetMessageType() const { return m_messageType; }
  inline bool MessageTypeHasBeenSet() const { return m_messageTypeHasBeenSet; }
  inline void SetMessageType(MessageType value) {
    m_messageTypeHasBeenSet = true;
    m_messageType = value;
  }
  inline PoolInformation& WithMessageType(MessageType value) {
    SetMessageType(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>When set to true you can receive incoming text messages from your end
   * recipients using the TwoWayChannelArn.</p>
   */
  inline bool GetTwoWayEnabled() const { return m_twoWayEnabled; }
  inline bool TwoWayEnabledHasBeenSet() const { return m_twoWayEnabledHasBeenSet; }
  inline void SetTwoWayEnabled(bool value) {
    m_twoWayEnabledHasBeenSet = true;
    m_twoWayEnabled = value;
  }
  inline PoolInformation& WithTwoWayEnabled(bool value) {
    SetTwoWayEnabled(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the two way channel.</p>
   */
  inline const Aws::String& GetTwoWayChannelArn() const { return m_twoWayChannelArn; }
  inline bool TwoWayChannelArnHasBeenSet() const { return m_twoWayChannelArnHasBeenSet; }
  template <typename TwoWayChannelArnT = Aws::String>
  void SetTwoWayChannelArn(TwoWayChannelArnT&& value) {
    m_twoWayChannelArnHasBeenSet = true;
    m_twoWayChannelArn = std::forward<TwoWayChannelArnT>(value);
  }
  template <typename TwoWayChannelArnT = Aws::String>
  PoolInformation& WithTwoWayChannelArn(TwoWayChannelArnT&& value) {
    SetTwoWayChannelArn(std::forward<TwoWayChannelArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>An optional IAM Role Arn for a service to assume, to be able to post inbound
   * SMS messages.</p>
   */
  inline const Aws::String& GetTwoWayChannelRole() const { return m_twoWayChannelRole; }
  inline bool TwoWayChannelRoleHasBeenSet() const { return m_twoWayChannelRoleHasBeenSet; }
  template <typename TwoWayChannelRoleT = Aws::String>
  void SetTwoWayChannelRole(TwoWayChannelRoleT&& value) {
    m_twoWayChannelRoleHasBeenSet = true;
    m_twoWayChannelRole = std::forward<TwoWayChannelRoleT>(value);
  }
  template <typename TwoWayChannelRoleT = Aws::String>
  PoolInformation& WithTwoWayChannelRole(TwoWayChannelRoleT&& value) {
    SetTwoWayChannelRole(std::forward<TwoWayChannelRoleT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>When set to false, an end recipient sends a message that begins with HELP or
   * STOP to one of your dedicated numbers, AWS End User Messaging SMS and Voice
   * automatically replies with a customizable message and adds the end recipient to
   * the OptOutList. When set to true you're responsible for responding to HELP and
   * STOP requests. You're also responsible for tracking and honoring opt-out
   * requests. For more information see <a
   * href="https://docs.aws.amazon.com/pinpoint/latest/userguide/settings-sms-managing.html#settings-account-sms-self-managed-opt-out">Self-managed
   * opt-outs</a> </p>
   */
  inline bool GetSelfManagedOptOutsEnabled() const { return m_selfManagedOptOutsEnabled; }
  inline bool SelfManagedOptOutsEnabledHasBeenSet() const { return m_selfManagedOptOutsEnabledHasBeenSet; }
  inline void SetSelfManagedOptOutsEnabled(bool value) {
    m_selfManagedOptOutsEnabledHasBeenSet = true;
    m_selfManagedOptOutsEnabled = value;
  }
  inline PoolInformation& WithSelfManagedOptOutsEnabled(bool value) {
    SetSelfManagedOptOutsEnabled(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the OptOutList associated with the pool.</p>
   */
  inline const Aws::String& GetOptOutListName() const { return m_optOutListName; }
  inline bool OptOutListNameHasBeenSet() const { return m_optOutListNameHasBeenSet; }
  template <typename OptOutListNameT = Aws::String>
  void SetOptOutListName(OptOutListNameT&& value) {
    m_optOutListNameHasBeenSet = true;
    m_optOutListName = std::forward<OptOutListNameT>(value);
  }
  template <typename OptOutListNameT = Aws::String>
  PoolInformation& WithOptOutListName(OptOutListNameT&& value) {
    SetOptOutListName(std::forward<OptOutListNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Allows you to enable shared routes on your pool.</p> <p>By default, this is
   * set to <code>False</code>. If you set this value to <code>True</code>, your
   * messages are sent using phone numbers or sender IDs (depending on the country)
   * that are shared with other users. In some countries, such as the United States,
   * senders aren't allowed to use shared routes and must use a dedicated phone
   * number or short code.</p>
   */
  inline bool GetSharedRoutesEnabled() const { return m_sharedRoutesEnabled; }
  inline bool SharedRoutesEnabledHasBeenSet() const { return m_sharedRoutesEnabledHasBeenSet; }
  inline void SetSharedRoutesEnabled(bool value) {
    m_sharedRoutesEnabledHasBeenSet = true;
    m_sharedRoutesEnabled = value;
  }
  inline PoolInformation& WithSharedRoutesEnabled(bool value) {
    SetSharedRoutesEnabled(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>When set to true the pool can't be deleted.</p>
   */
  inline bool GetDeletionProtectionEnabled() const { return m_deletionProtectionEnabled; }
  inline bool DeletionProtectionEnabledHasBeenSet() const { return m_deletionProtectionEnabledHasBeenSet; }
  inline void SetDeletionProtectionEnabled(bool value) {
    m_deletionProtectionEnabledHasBeenSet = true;
    m_deletionProtectionEnabled = value;
  }
  inline PoolInformation& WithDeletionProtectionEnabled(bool value) {
    SetDeletionProtectionEnabled(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The time when the pool was created, in <a
   * href="https://www.epochconverter.com/">UNIX epoch time</a> format.</p>
   */
  inline const Aws::Utils::DateTime& GetCreatedTimestamp() const { return m_createdTimestamp; }
  inline bool CreatedTimestampHasBeenSet() const { return m_createdTimestampHasBeenSet; }
  template <typename CreatedTimestampT = Aws::Utils::DateTime>
  void SetCreatedTimestamp(CreatedTimestampT&& value) {
    m_createdTimestampHasBeenSet = true;
    m_createdTimestamp = std::forward<CreatedTimestampT>(value);
  }
  template <typename CreatedTimestampT = Aws::Utils::DateTime>
  PoolInformation& WithCreatedTimestamp(CreatedTimestampT&& value) {
    SetCreatedTimestamp(std::forward<CreatedTimestampT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_poolArn;
  bool m_poolArnHasBeenSet = false;

  Aws::String m_poolId;
  bool m_poolIdHasBeenSet = false;

  PoolStatus m_status{PoolStatus::NOT_SET};
  bool m_statusHasBeenSet = false;

  MessageType m_messageType{MessageType::NOT_SET};
  bool m_messageTypeHasBeenSet = false;

  bool m_twoWayEnabled{false};
  bool m_twoWayEnabledHasBeenSet = false;

  Aws::String m_twoWayChannelArn;
  bool m_twoWayChannelArnHasBeenSet = false;

  Aws::String m_twoWayChannelRole;
  bool m_twoWayChannelRoleHasBeenSet = false;

  bool m_selfManagedOptOutsEnabled{false};
  bool m_selfManagedOptOutsEnabledHasBeenSet = false;

  Aws::String m_optOutListName;
  bool m_optOutListNameHasBeenSet = false;

  bool m_sharedRoutesEnabled{false};
  bool m_sharedRoutesEnabledHasBeenSet = false;

  bool m_deletionProtectionEnabled{false};
  bool m_deletionProtectionEnabledHasBeenSet = false;

  Aws::Utils::DateTime m_createdTimestamp{};
  bool m_createdTimestampHasBeenSet = false;
};

}  // namespace Model
}  // namespace PinpointSMSVoiceV2
}  // namespace Aws
