﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/appfabric/AppFabricRequest.h>
#include <aws/appfabric/AppFabric_EXPORTS.h>
#include <aws/appfabric/model/Tag.h>
#include <aws/core/utils/UUID.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>

#include <utility>

namespace Aws {
namespace AppFabric {
namespace Model {

/**
 */
class CreateAppBundleRequest : public AppFabricRequest {
 public:
  AWS_APPFABRIC_API CreateAppBundleRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateAppBundle"; }

  AWS_APPFABRIC_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>Specifies a unique, case-sensitive identifier that you provide to ensure the
   * idempotency of the request. This lets you safely retry the request without
   * accidentally performing the same operation a second time. Passing the same value
   * to a later call to an operation requires that you also pass the same value for
   * all other parameters. We recommend that you use a <a
   * href="https://wikipedia.org/wiki/Universally_unique_identifier">UUID type of
   * value</a>.</p> <p>If you don't provide this value, then Amazon Web Services
   * generates a random one for you.</p> <p>If you retry the operation with the same
   * <code>ClientToken</code>, but with different parameters, the retry fails with an
   * <code>IdempotentParameterMismatch</code> error.</p>
   */
  inline const Aws::String& GetClientToken() const { return m_clientToken; }
  inline bool ClientTokenHasBeenSet() const { return m_clientTokenHasBeenSet; }
  template <typename ClientTokenT = Aws::String>
  void SetClientToken(ClientTokenT&& value) {
    m_clientTokenHasBeenSet = true;
    m_clientToken = std::forward<ClientTokenT>(value);
  }
  template <typename ClientTokenT = Aws::String>
  CreateAppBundleRequest& WithClientToken(ClientTokenT&& value) {
    SetClientToken(std::forward<ClientTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the Key Management Service (KMS) key to use
   * to encrypt the application data. If this is not specified, an Amazon Web
   * Services owned key is used for encryption.</p>
   */
  inline const Aws::String& GetCustomerManagedKeyIdentifier() const { return m_customerManagedKeyIdentifier; }
  inline bool CustomerManagedKeyIdentifierHasBeenSet() const { return m_customerManagedKeyIdentifierHasBeenSet; }
  template <typename CustomerManagedKeyIdentifierT = Aws::String>
  void SetCustomerManagedKeyIdentifier(CustomerManagedKeyIdentifierT&& value) {
    m_customerManagedKeyIdentifierHasBeenSet = true;
    m_customerManagedKeyIdentifier = std::forward<CustomerManagedKeyIdentifierT>(value);
  }
  template <typename CustomerManagedKeyIdentifierT = Aws::String>
  CreateAppBundleRequest& WithCustomerManagedKeyIdentifier(CustomerManagedKeyIdentifierT&& value) {
    SetCustomerManagedKeyIdentifier(std::forward<CustomerManagedKeyIdentifierT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A map of the key-value pairs of the tag or tags to assign to the
   * resource.</p>
   */
  inline const Aws::Vector<Tag>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Vector<Tag>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Vector<Tag>>
  CreateAppBundleRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsT = Tag>
  CreateAppBundleRequest& AddTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace_back(std::forward<TagsT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_clientToken{Aws::Utils::UUID::PseudoRandomUUID()};
  bool m_clientTokenHasBeenSet = true;

  Aws::String m_customerManagedKeyIdentifier;
  bool m_customerManagedKeyIdentifierHasBeenSet = false;

  Aws::Vector<Tag> m_tags;
  bool m_tagsHasBeenSet = false;
};

}  // namespace Model
}  // namespace AppFabric
}  // namespace Aws
