﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/frauddetector/FraudDetectorRequest.h>
#include <aws/frauddetector/FraudDetector_EXPORTS.h>
#include <aws/frauddetector/model/ModelTypeEnum.h>

#include <utility>

namespace Aws {
namespace FraudDetector {
namespace Model {

/**
 */
class DescribeModelVersionsRequest : public FraudDetectorRequest {
 public:
  AWS_FRAUDDETECTOR_API DescribeModelVersionsRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "DescribeModelVersions"; }

  AWS_FRAUDDETECTOR_API Aws::String SerializePayload() const override;

  AWS_FRAUDDETECTOR_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The model ID.</p>
   */
  inline const Aws::String& GetModelId() const { return m_modelId; }
  inline bool ModelIdHasBeenSet() const { return m_modelIdHasBeenSet; }
  template <typename ModelIdT = Aws::String>
  void SetModelId(ModelIdT&& value) {
    m_modelIdHasBeenSet = true;
    m_modelId = std::forward<ModelIdT>(value);
  }
  template <typename ModelIdT = Aws::String>
  DescribeModelVersionsRequest& WithModelId(ModelIdT&& value) {
    SetModelId(std::forward<ModelIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The model version number.</p>
   */
  inline const Aws::String& GetModelVersionNumber() const { return m_modelVersionNumber; }
  inline bool ModelVersionNumberHasBeenSet() const { return m_modelVersionNumberHasBeenSet; }
  template <typename ModelVersionNumberT = Aws::String>
  void SetModelVersionNumber(ModelVersionNumberT&& value) {
    m_modelVersionNumberHasBeenSet = true;
    m_modelVersionNumber = std::forward<ModelVersionNumberT>(value);
  }
  template <typename ModelVersionNumberT = Aws::String>
  DescribeModelVersionsRequest& WithModelVersionNumber(ModelVersionNumberT&& value) {
    SetModelVersionNumber(std::forward<ModelVersionNumberT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The model type.</p>
   */
  inline ModelTypeEnum GetModelType() const { return m_modelType; }
  inline bool ModelTypeHasBeenSet() const { return m_modelTypeHasBeenSet; }
  inline void SetModelType(ModelTypeEnum value) {
    m_modelTypeHasBeenSet = true;
    m_modelType = value;
  }
  inline DescribeModelVersionsRequest& WithModelType(ModelTypeEnum value) {
    SetModelType(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The next token from the previous results.</p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  DescribeModelVersionsRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of results to return.</p>
   */
  inline int GetMaxResults() const { return m_maxResults; }
  inline bool MaxResultsHasBeenSet() const { return m_maxResultsHasBeenSet; }
  inline void SetMaxResults(int value) {
    m_maxResultsHasBeenSet = true;
    m_maxResults = value;
  }
  inline DescribeModelVersionsRequest& WithMaxResults(int value) {
    SetMaxResults(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_modelId;
  bool m_modelIdHasBeenSet = false;

  Aws::String m_modelVersionNumber;
  bool m_modelVersionNumberHasBeenSet = false;

  ModelTypeEnum m_modelType{ModelTypeEnum::NOT_SET};
  bool m_modelTypeHasBeenSet = false;

  Aws::String m_nextToken;
  bool m_nextTokenHasBeenSet = false;

  int m_maxResults{0};
  bool m_maxResultsHasBeenSet = false;
};

}  // namespace Model
}  // namespace FraudDetector
}  // namespace Aws
