﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/ivs-realtime/IvsrealtimeRequest.h>
#include <aws/ivs-realtime/Ivsrealtime_EXPORTS.h>
#include <aws/ivs-realtime/model/IngestProtocol.h>

#include <utility>

namespace Aws {
namespace ivsrealtime {
namespace Model {

/**
 */
class CreateIngestConfigurationRequest : public IvsrealtimeRequest {
 public:
  AWS_IVSREALTIME_API CreateIngestConfigurationRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateIngestConfiguration"; }

  AWS_IVSREALTIME_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>Optional name that can be specified for the IngestConfiguration being
   * created.</p>
   */
  inline const Aws::String& GetName() const { return m_name; }
  inline bool NameHasBeenSet() const { return m_nameHasBeenSet; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  CreateIngestConfigurationRequest& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>ARN of the stage with which the IngestConfiguration is associated.</p>
   */
  inline const Aws::String& GetStageArn() const { return m_stageArn; }
  inline bool StageArnHasBeenSet() const { return m_stageArnHasBeenSet; }
  template <typename StageArnT = Aws::String>
  void SetStageArn(StageArnT&& value) {
    m_stageArnHasBeenSet = true;
    m_stageArn = std::forward<StageArnT>(value);
  }
  template <typename StageArnT = Aws::String>
  CreateIngestConfigurationRequest& WithStageArn(StageArnT&& value) {
    SetStageArn(std::forward<StageArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Customer-assigned name to help identify the participant using the
   * IngestConfiguration; this can be used to link a participant to a user in the
   * customer’s own systems. This can be any UTF-8 encoded text. <i>This field is
   * exposed to all stage participants and should not be used for personally
   * identifying, confidential, or sensitive information.</i> </p>
   */
  inline const Aws::String& GetUserId() const { return m_userId; }
  inline bool UserIdHasBeenSet() const { return m_userIdHasBeenSet; }
  template <typename UserIdT = Aws::String>
  void SetUserId(UserIdT&& value) {
    m_userIdHasBeenSet = true;
    m_userId = std::forward<UserIdT>(value);
  }
  template <typename UserIdT = Aws::String>
  CreateIngestConfigurationRequest& WithUserId(UserIdT&& value) {
    SetUserId(std::forward<UserIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Application-provided attributes to store in the IngestConfiguration and
   * attach to a stage. Map keys and values can contain UTF-8 encoded text. The
   * maximum length of this field is 1 KB total. <i>This field is exposed to all
   * stage participants and should not be used for personally identifying,
   * confidential, or sensitive information.</i> </p>
   */
  inline const Aws::Map<Aws::String, Aws::String>& GetAttributes() const { return m_attributes; }
  inline bool AttributesHasBeenSet() const { return m_attributesHasBeenSet; }
  template <typename AttributesT = Aws::Map<Aws::String, Aws::String>>
  void SetAttributes(AttributesT&& value) {
    m_attributesHasBeenSet = true;
    m_attributes = std::forward<AttributesT>(value);
  }
  template <typename AttributesT = Aws::Map<Aws::String, Aws::String>>
  CreateIngestConfigurationRequest& WithAttributes(AttributesT&& value) {
    SetAttributes(std::forward<AttributesT>(value));
    return *this;
  }
  template <typename AttributesKeyT = Aws::String, typename AttributesValueT = Aws::String>
  CreateIngestConfigurationRequest& AddAttributes(AttributesKeyT&& key, AttributesValueT&& value) {
    m_attributesHasBeenSet = true;
    m_attributes.emplace(std::forward<AttributesKeyT>(key), std::forward<AttributesValueT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Type of ingest protocol that the user employs to broadcast. If this is set to
   * <code>RTMP</code>, <code>insecureIngest</code> must be set to
   * <code>true</code>.</p>
   */
  inline IngestProtocol GetIngestProtocol() const { return m_ingestProtocol; }
  inline bool IngestProtocolHasBeenSet() const { return m_ingestProtocolHasBeenSet; }
  inline void SetIngestProtocol(IngestProtocol value) {
    m_ingestProtocolHasBeenSet = true;
    m_ingestProtocol = value;
  }
  inline CreateIngestConfigurationRequest& WithIngestProtocol(IngestProtocol value) {
    SetIngestProtocol(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Whether the stage allows insecure RTMP ingest. This must be set to
   * <code>true</code>, if <code>ingestProtocol</code> is set to <code>RTMP</code>.
   * Default: <code>false</code>. </p>
   */
  inline bool GetInsecureIngest() const { return m_insecureIngest; }
  inline bool InsecureIngestHasBeenSet() const { return m_insecureIngestHasBeenSet; }
  inline void SetInsecureIngest(bool value) {
    m_insecureIngestHasBeenSet = true;
    m_insecureIngest = value;
  }
  inline CreateIngestConfigurationRequest& WithInsecureIngest(bool value) {
    SetInsecureIngest(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Tags attached to the resource. Array of maps, each of the form
   * <code>string:string (key:value)</code>. See <a
   * href="https://docs.aws.amazon.com/tag-editor/latest/userguide/best-practices-and-strats.html">Best
   * practices and strategies</a> in <i>Tagging AWS Resources and Tag Editor</i> for
   * details, including restrictions that apply to tags and "Tag naming limits and
   * requirements"; Amazon IVS has no constraints on tags beyond what is documented
   * there.</p>
   */
  inline const Aws::Map<Aws::String, Aws::String>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  CreateIngestConfigurationRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsKeyT = Aws::String, typename TagsValueT = Aws::String>
  CreateIngestConfigurationRequest& AddTags(TagsKeyT&& key, TagsValueT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace(std::forward<TagsKeyT>(key), std::forward<TagsValueT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_name;
  bool m_nameHasBeenSet = false;

  Aws::String m_stageArn;
  bool m_stageArnHasBeenSet = false;

  Aws::String m_userId;
  bool m_userIdHasBeenSet = false;

  Aws::Map<Aws::String, Aws::String> m_attributes;
  bool m_attributesHasBeenSet = false;

  IngestProtocol m_ingestProtocol{IngestProtocol::NOT_SET};
  bool m_ingestProtocolHasBeenSet = false;

  bool m_insecureIngest{false};
  bool m_insecureIngestHasBeenSet = false;

  Aws::Map<Aws::String, Aws::String> m_tags;
  bool m_tagsHasBeenSet = false;
};

}  // namespace Model
}  // namespace ivsrealtime
}  // namespace Aws
