﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/UUID.h>
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/medialive/MediaLiveRequest.h>
#include <aws/medialive/MediaLive_EXPORTS.h>

#include <utility>

namespace Aws {
namespace MediaLive {
namespace Model {

/**
 * A request to create a channel placement group.<p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/medialive-2017-10-14/CreateChannelPlacementGroupRequest">AWS
 * API Reference</a></p>
 */
class CreateChannelPlacementGroupRequest : public MediaLiveRequest {
 public:
  AWS_MEDIALIVE_API CreateChannelPlacementGroupRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateChannelPlacementGroup"; }

  AWS_MEDIALIVE_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * The ID of the cluster.
   */
  inline const Aws::String& GetClusterId() const { return m_clusterId; }
  inline bool ClusterIdHasBeenSet() const { return m_clusterIdHasBeenSet; }
  template <typename ClusterIdT = Aws::String>
  void SetClusterId(ClusterIdT&& value) {
    m_clusterIdHasBeenSet = true;
    m_clusterId = std::forward<ClusterIdT>(value);
  }
  template <typename ClusterIdT = Aws::String>
  CreateChannelPlacementGroupRequest& WithClusterId(ClusterIdT&& value) {
    SetClusterId(std::forward<ClusterIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * Specify a name that is unique in the Cluster. You can't change the name. Names
   * are case-sensitive.
   */
  inline const Aws::String& GetName() const { return m_name; }
  inline bool NameHasBeenSet() const { return m_nameHasBeenSet; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  CreateChannelPlacementGroupRequest& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * An array of one ID for the Node that you want to associate with the
   * ChannelPlacementGroup. (You can't associate more than one Node with the
   * ChannelPlacementGroup.) The Node and the ChannelPlacementGroup must be in the
   * same Cluster.
   */
  inline const Aws::Vector<Aws::String>& GetNodes() const { return m_nodes; }
  inline bool NodesHasBeenSet() const { return m_nodesHasBeenSet; }
  template <typename NodesT = Aws::Vector<Aws::String>>
  void SetNodes(NodesT&& value) {
    m_nodesHasBeenSet = true;
    m_nodes = std::forward<NodesT>(value);
  }
  template <typename NodesT = Aws::Vector<Aws::String>>
  CreateChannelPlacementGroupRequest& WithNodes(NodesT&& value) {
    SetNodes(std::forward<NodesT>(value));
    return *this;
  }
  template <typename NodesT = Aws::String>
  CreateChannelPlacementGroupRequest& AddNodes(NodesT&& value) {
    m_nodesHasBeenSet = true;
    m_nodes.emplace_back(std::forward<NodesT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * An ID that you assign to a create request. This ID ensures idempotency when
   * creating resources. the request.
   */
  inline const Aws::String& GetRequestId() const { return m_requestId; }
  inline bool RequestIdHasBeenSet() const { return m_requestIdHasBeenSet; }
  template <typename RequestIdT = Aws::String>
  void SetRequestId(RequestIdT&& value) {
    m_requestIdHasBeenSet = true;
    m_requestId = std::forward<RequestIdT>(value);
  }
  template <typename RequestIdT = Aws::String>
  CreateChannelPlacementGroupRequest& WithRequestId(RequestIdT&& value) {
    SetRequestId(std::forward<RequestIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * A collection of key-value pairs.
   */
  inline const Aws::Map<Aws::String, Aws::String>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  CreateChannelPlacementGroupRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsKeyT = Aws::String, typename TagsValueT = Aws::String>
  CreateChannelPlacementGroupRequest& AddTags(TagsKeyT&& key, TagsValueT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace(std::forward<TagsKeyT>(key), std::forward<TagsValueT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_clusterId;
  bool m_clusterIdHasBeenSet = false;

  Aws::String m_name;
  bool m_nameHasBeenSet = false;

  Aws::Vector<Aws::String> m_nodes;
  bool m_nodesHasBeenSet = false;

  Aws::String m_requestId{Aws::Utils::UUID::PseudoRandomUUID()};
  bool m_requestIdHasBeenSet = true;

  Aws::Map<Aws::String, Aws::String> m_tags;
  bool m_tagsHasBeenSet = false;
};

}  // namespace Model
}  // namespace MediaLive
}  // namespace Aws
