﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/network-firewall/NetworkFirewallRequest.h>
#include <aws/network-firewall/NetworkFirewall_EXPORTS.h>

#include <utility>

namespace Aws {
namespace NetworkFirewall {
namespace Model {

/**
 */
class DescribeFlowOperationRequest : public NetworkFirewallRequest {
 public:
  AWS_NETWORKFIREWALL_API DescribeFlowOperationRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "DescribeFlowOperation"; }

  AWS_NETWORKFIREWALL_API Aws::String SerializePayload() const override;

  AWS_NETWORKFIREWALL_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the firewall.</p>
   */
  inline const Aws::String& GetFirewallArn() const { return m_firewallArn; }
  inline bool FirewallArnHasBeenSet() const { return m_firewallArnHasBeenSet; }
  template <typename FirewallArnT = Aws::String>
  void SetFirewallArn(FirewallArnT&& value) {
    m_firewallArnHasBeenSet = true;
    m_firewallArn = std::forward<FirewallArnT>(value);
  }
  template <typename FirewallArnT = Aws::String>
  DescribeFlowOperationRequest& WithFirewallArn(FirewallArnT&& value) {
    SetFirewallArn(std::forward<FirewallArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of the Availability Zone where the firewall is located. For example,
   * <code>us-east-2a</code>.</p> <p>Defines the scope a flow operation. You can use
   * up to 20 filters to configure a single flow operation.</p>
   */
  inline const Aws::String& GetAvailabilityZone() const { return m_availabilityZone; }
  inline bool AvailabilityZoneHasBeenSet() const { return m_availabilityZoneHasBeenSet; }
  template <typename AvailabilityZoneT = Aws::String>
  void SetAvailabilityZone(AvailabilityZoneT&& value) {
    m_availabilityZoneHasBeenSet = true;
    m_availabilityZone = std::forward<AvailabilityZoneT>(value);
  }
  template <typename AvailabilityZoneT = Aws::String>
  DescribeFlowOperationRequest& WithAvailabilityZone(AvailabilityZoneT&& value) {
    SetAvailabilityZone(std::forward<AvailabilityZoneT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of a VPC endpoint association.</p>
   */
  inline const Aws::String& GetVpcEndpointAssociationArn() const { return m_vpcEndpointAssociationArn; }
  inline bool VpcEndpointAssociationArnHasBeenSet() const { return m_vpcEndpointAssociationArnHasBeenSet; }
  template <typename VpcEndpointAssociationArnT = Aws::String>
  void SetVpcEndpointAssociationArn(VpcEndpointAssociationArnT&& value) {
    m_vpcEndpointAssociationArnHasBeenSet = true;
    m_vpcEndpointAssociationArn = std::forward<VpcEndpointAssociationArnT>(value);
  }
  template <typename VpcEndpointAssociationArnT = Aws::String>
  DescribeFlowOperationRequest& WithVpcEndpointAssociationArn(VpcEndpointAssociationArnT&& value) {
    SetVpcEndpointAssociationArn(std::forward<VpcEndpointAssociationArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A unique identifier for the primary endpoint associated with a firewall.</p>
   */
  inline const Aws::String& GetVpcEndpointId() const { return m_vpcEndpointId; }
  inline bool VpcEndpointIdHasBeenSet() const { return m_vpcEndpointIdHasBeenSet; }
  template <typename VpcEndpointIdT = Aws::String>
  void SetVpcEndpointId(VpcEndpointIdT&& value) {
    m_vpcEndpointIdHasBeenSet = true;
    m_vpcEndpointId = std::forward<VpcEndpointIdT>(value);
  }
  template <typename VpcEndpointIdT = Aws::String>
  DescribeFlowOperationRequest& WithVpcEndpointId(VpcEndpointIdT&& value) {
    SetVpcEndpointId(std::forward<VpcEndpointIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A unique identifier for the flow operation. This ID is returned in the
   * responses to start and list commands. You provide to describe commands.</p>
   */
  inline const Aws::String& GetFlowOperationId() const { return m_flowOperationId; }
  inline bool FlowOperationIdHasBeenSet() const { return m_flowOperationIdHasBeenSet; }
  template <typename FlowOperationIdT = Aws::String>
  void SetFlowOperationId(FlowOperationIdT&& value) {
    m_flowOperationIdHasBeenSet = true;
    m_flowOperationId = std::forward<FlowOperationIdT>(value);
  }
  template <typename FlowOperationIdT = Aws::String>
  DescribeFlowOperationRequest& WithFlowOperationId(FlowOperationIdT&& value) {
    SetFlowOperationId(std::forward<FlowOperationIdT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_firewallArn;
  bool m_firewallArnHasBeenSet = false;

  Aws::String m_availabilityZone;
  bool m_availabilityZoneHasBeenSet = false;

  Aws::String m_vpcEndpointAssociationArn;
  bool m_vpcEndpointAssociationArnHasBeenSet = false;

  Aws::String m_vpcEndpointId;
  bool m_vpcEndpointIdHasBeenSet = false;

  Aws::String m_flowOperationId;
  bool m_flowOperationIdHasBeenSet = false;
};

}  // namespace Model
}  // namespace NetworkFirewall
}  // namespace Aws
