﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/DateTime.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/osis/OSIS_EXPORTS.h>
#include <aws/osis/model/PipelineDestination.h>
#include <aws/osis/model/PipelineStatus.h>
#include <aws/osis/model/PipelineStatusReason.h>
#include <aws/osis/model/Tag.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace OSIS {
namespace Model {

/**
 * <p>Summary information for an OpenSearch Ingestion pipeline.</p><p><h3>See
 * Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/osis-2022-01-01/PipelineSummary">AWS
 * API Reference</a></p>
 */
class PipelineSummary {
 public:
  AWS_OSIS_API PipelineSummary() = default;
  AWS_OSIS_API PipelineSummary(Aws::Utils::Json::JsonView jsonValue);
  AWS_OSIS_API PipelineSummary& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_OSIS_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>The current status of the pipeline.</p>
   */
  inline PipelineStatus GetStatus() const { return m_status; }
  inline bool StatusHasBeenSet() const { return m_statusHasBeenSet; }
  inline void SetStatus(PipelineStatus value) {
    m_statusHasBeenSet = true;
    m_status = value;
  }
  inline PipelineSummary& WithStatus(PipelineStatus value) {
    SetStatus(value);
    return *this;
  }
  ///@}

  ///@{

  inline const PipelineStatusReason& GetStatusReason() const { return m_statusReason; }
  inline bool StatusReasonHasBeenSet() const { return m_statusReasonHasBeenSet; }
  template <typename StatusReasonT = PipelineStatusReason>
  void SetStatusReason(StatusReasonT&& value) {
    m_statusReasonHasBeenSet = true;
    m_statusReason = std::forward<StatusReasonT>(value);
  }
  template <typename StatusReasonT = PipelineStatusReason>
  PipelineSummary& WithStatusReason(StatusReasonT&& value) {
    SetStatusReason(std::forward<StatusReasonT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the pipeline.</p>
   */
  inline const Aws::String& GetPipelineName() const { return m_pipelineName; }
  inline bool PipelineNameHasBeenSet() const { return m_pipelineNameHasBeenSet; }
  template <typename PipelineNameT = Aws::String>
  void SetPipelineName(PipelineNameT&& value) {
    m_pipelineNameHasBeenSet = true;
    m_pipelineName = std::forward<PipelineNameT>(value);
  }
  template <typename PipelineNameT = Aws::String>
  PipelineSummary& WithPipelineName(PipelineNameT&& value) {
    SetPipelineName(std::forward<PipelineNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the pipeline.</p>
   */
  inline const Aws::String& GetPipelineArn() const { return m_pipelineArn; }
  inline bool PipelineArnHasBeenSet() const { return m_pipelineArnHasBeenSet; }
  template <typename PipelineArnT = Aws::String>
  void SetPipelineArn(PipelineArnT&& value) {
    m_pipelineArnHasBeenSet = true;
    m_pipelineArn = std::forward<PipelineArnT>(value);
  }
  template <typename PipelineArnT = Aws::String>
  PipelineSummary& WithPipelineArn(PipelineArnT&& value) {
    SetPipelineArn(std::forward<PipelineArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The minimum pipeline capacity, in Ingestion Compute Units (ICUs).</p>
   */
  inline int GetMinUnits() const { return m_minUnits; }
  inline bool MinUnitsHasBeenSet() const { return m_minUnitsHasBeenSet; }
  inline void SetMinUnits(int value) {
    m_minUnitsHasBeenSet = true;
    m_minUnits = value;
  }
  inline PipelineSummary& WithMinUnits(int value) {
    SetMinUnits(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum pipeline capacity, in Ingestion Compute Units (ICUs).</p>
   */
  inline int GetMaxUnits() const { return m_maxUnits; }
  inline bool MaxUnitsHasBeenSet() const { return m_maxUnitsHasBeenSet; }
  inline void SetMaxUnits(int value) {
    m_maxUnitsHasBeenSet = true;
    m_maxUnits = value;
  }
  inline PipelineSummary& WithMaxUnits(int value) {
    SetMaxUnits(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The date and time when the pipeline was created.</p>
   */
  inline const Aws::Utils::DateTime& GetCreatedAt() const { return m_createdAt; }
  inline bool CreatedAtHasBeenSet() const { return m_createdAtHasBeenSet; }
  template <typename CreatedAtT = Aws::Utils::DateTime>
  void SetCreatedAt(CreatedAtT&& value) {
    m_createdAtHasBeenSet = true;
    m_createdAt = std::forward<CreatedAtT>(value);
  }
  template <typename CreatedAtT = Aws::Utils::DateTime>
  PipelineSummary& WithCreatedAt(CreatedAtT&& value) {
    SetCreatedAt(std::forward<CreatedAtT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The date and time when the pipeline was last updated.</p>
   */
  inline const Aws::Utils::DateTime& GetLastUpdatedAt() const { return m_lastUpdatedAt; }
  inline bool LastUpdatedAtHasBeenSet() const { return m_lastUpdatedAtHasBeenSet; }
  template <typename LastUpdatedAtT = Aws::Utils::DateTime>
  void SetLastUpdatedAt(LastUpdatedAtT&& value) {
    m_lastUpdatedAtHasBeenSet = true;
    m_lastUpdatedAt = std::forward<LastUpdatedAtT>(value);
  }
  template <typename LastUpdatedAtT = Aws::Utils::DateTime>
  PipelineSummary& WithLastUpdatedAt(LastUpdatedAtT&& value) {
    SetLastUpdatedAt(std::forward<LastUpdatedAtT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A list of destinations to which the pipeline writes data.</p>
   */
  inline const Aws::Vector<PipelineDestination>& GetDestinations() const { return m_destinations; }
  inline bool DestinationsHasBeenSet() const { return m_destinationsHasBeenSet; }
  template <typename DestinationsT = Aws::Vector<PipelineDestination>>
  void SetDestinations(DestinationsT&& value) {
    m_destinationsHasBeenSet = true;
    m_destinations = std::forward<DestinationsT>(value);
  }
  template <typename DestinationsT = Aws::Vector<PipelineDestination>>
  PipelineSummary& WithDestinations(DestinationsT&& value) {
    SetDestinations(std::forward<DestinationsT>(value));
    return *this;
  }
  template <typename DestinationsT = PipelineDestination>
  PipelineSummary& AddDestinations(DestinationsT&& value) {
    m_destinationsHasBeenSet = true;
    m_destinations.emplace_back(std::forward<DestinationsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A list of tags associated with the given pipeline.</p>
   */
  inline const Aws::Vector<Tag>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Vector<Tag>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Vector<Tag>>
  PipelineSummary& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsT = Tag>
  PipelineSummary& AddTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace_back(std::forward<TagsT>(value));
    return *this;
  }
  ///@}
 private:
  PipelineStatus m_status{PipelineStatus::NOT_SET};
  bool m_statusHasBeenSet = false;

  PipelineStatusReason m_statusReason;
  bool m_statusReasonHasBeenSet = false;

  Aws::String m_pipelineName;
  bool m_pipelineNameHasBeenSet = false;

  Aws::String m_pipelineArn;
  bool m_pipelineArnHasBeenSet = false;

  int m_minUnits{0};
  bool m_minUnitsHasBeenSet = false;

  int m_maxUnits{0};
  bool m_maxUnitsHasBeenSet = false;

  Aws::Utils::DateTime m_createdAt{};
  bool m_createdAtHasBeenSet = false;

  Aws::Utils::DateTime m_lastUpdatedAt{};
  bool m_lastUpdatedAtHasBeenSet = false;

  Aws::Vector<PipelineDestination> m_destinations;
  bool m_destinationsHasBeenSet = false;

  Aws::Vector<Tag> m_tags;
  bool m_tagsHasBeenSet = false;
};

}  // namespace Model
}  // namespace OSIS
}  // namespace Aws
