﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/waf/WAFRequest.h>
#include <aws/waf/WAF_EXPORTS.h>

#include <utility>

namespace Aws {
namespace WAF {
namespace Model {

/**
 */
class DeleteWebACLRequest : public WAFRequest {
 public:
  AWS_WAF_API DeleteWebACLRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "DeleteWebACL"; }

  AWS_WAF_API Aws::String SerializePayload() const override;

  AWS_WAF_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The <code>WebACLId</code> of the <a>WebACL</a> that you want to delete.
   * <code>WebACLId</code> is returned by <a>CreateWebACL</a> and by
   * <a>ListWebACLs</a>.</p>
   */
  inline const Aws::String& GetWebACLId() const { return m_webACLId; }
  inline bool WebACLIdHasBeenSet() const { return m_webACLIdHasBeenSet; }
  template <typename WebACLIdT = Aws::String>
  void SetWebACLId(WebACLIdT&& value) {
    m_webACLIdHasBeenSet = true;
    m_webACLId = std::forward<WebACLIdT>(value);
  }
  template <typename WebACLIdT = Aws::String>
  DeleteWebACLRequest& WithWebACLId(WebACLIdT&& value) {
    SetWebACLId(std::forward<WebACLIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The value returned by the most recent call to <a>GetChangeToken</a>.</p>
   */
  inline const Aws::String& GetChangeToken() const { return m_changeToken; }
  inline bool ChangeTokenHasBeenSet() const { return m_changeTokenHasBeenSet; }
  template <typename ChangeTokenT = Aws::String>
  void SetChangeToken(ChangeTokenT&& value) {
    m_changeTokenHasBeenSet = true;
    m_changeToken = std::forward<ChangeTokenT>(value);
  }
  template <typename ChangeTokenT = Aws::String>
  DeleteWebACLRequest& WithChangeToken(ChangeTokenT&& value) {
    SetChangeToken(std::forward<ChangeTokenT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_webACLId;
  bool m_webACLIdHasBeenSet = false;

  Aws::String m_changeToken;
  bool m_changeTokenHasBeenSet = false;
};

}  // namespace Model
}  // namespace WAF
}  // namespace Aws
