﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/cloudformation/CloudFormationRequest.h>
#include <aws/cloudformation/CloudFormation_EXPORTS.h>
#include <aws/cloudformation/model/RegistrationStatus.h>
#include <aws/cloudformation/model/RegistryType.h>
#include <aws/core/utils/memory/stl/AWSString.h>

#include <utility>

namespace Aws {
namespace CloudFormation {
namespace Model {

/**
 */
class ListTypeRegistrationsRequest : public CloudFormationRequest {
 public:
  AWS_CLOUDFORMATION_API ListTypeRegistrationsRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ListTypeRegistrations"; }

  AWS_CLOUDFORMATION_API Aws::String SerializePayload() const override;

 protected:
  AWS_CLOUDFORMATION_API void DumpBodyToUrl(Aws::Http::URI& uri) const override;

 public:
  ///@{
  /**
   * <p>The kind of extension.</p> <p>Conditional: You must specify either
   * <code>TypeName</code> and <code>Type</code>, or <code>Arn</code>.</p>
   */
  inline RegistryType GetType() const { return m_type; }
  inline bool TypeHasBeenSet() const { return m_typeHasBeenSet; }
  inline void SetType(RegistryType value) {
    m_typeHasBeenSet = true;
    m_type = value;
  }
  inline ListTypeRegistrationsRequest& WithType(RegistryType value) {
    SetType(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the extension.</p> <p>Conditional: You must specify either
   * <code>TypeName</code> and <code>Type</code>, or <code>Arn</code>.</p>
   */
  inline const Aws::String& GetTypeName() const { return m_typeName; }
  inline bool TypeNameHasBeenSet() const { return m_typeNameHasBeenSet; }
  template <typename TypeNameT = Aws::String>
  void SetTypeName(TypeNameT&& value) {
    m_typeNameHasBeenSet = true;
    m_typeName = std::forward<TypeNameT>(value);
  }
  template <typename TypeNameT = Aws::String>
  ListTypeRegistrationsRequest& WithTypeName(TypeNameT&& value) {
    SetTypeName(std::forward<TypeNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the extension.</p> <p>Conditional: You must
   * specify either <code>TypeName</code> and <code>Type</code>, or
   * <code>Arn</code>.</p>
   */
  inline const Aws::String& GetTypeArn() const { return m_typeArn; }
  inline bool TypeArnHasBeenSet() const { return m_typeArnHasBeenSet; }
  template <typename TypeArnT = Aws::String>
  void SetTypeArn(TypeArnT&& value) {
    m_typeArnHasBeenSet = true;
    m_typeArn = std::forward<TypeArnT>(value);
  }
  template <typename TypeArnT = Aws::String>
  ListTypeRegistrationsRequest& WithTypeArn(TypeArnT&& value) {
    SetTypeArn(std::forward<TypeArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The current status of the extension registration request.</p> <p>The default
   * is <code>IN_PROGRESS</code>.</p>
   */
  inline RegistrationStatus GetRegistrationStatusFilter() const { return m_registrationStatusFilter; }
  inline bool RegistrationStatusFilterHasBeenSet() const { return m_registrationStatusFilterHasBeenSet; }
  inline void SetRegistrationStatusFilter(RegistrationStatus value) {
    m_registrationStatusFilterHasBeenSet = true;
    m_registrationStatusFilter = value;
  }
  inline ListTypeRegistrationsRequest& WithRegistrationStatusFilter(RegistrationStatus value) {
    SetRegistrationStatusFilter(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of results to be returned with a single call. If the
   * number of available results exceeds this maximum, the response includes a
   * <code>NextToken</code> value that you can assign to the <code>NextToken</code>
   * request parameter to get the next set of results.</p>
   */
  inline int GetMaxResults() const { return m_maxResults; }
  inline bool MaxResultsHasBeenSet() const { return m_maxResultsHasBeenSet; }
  inline void SetMaxResults(int value) {
    m_maxResultsHasBeenSet = true;
    m_maxResults = value;
  }
  inline ListTypeRegistrationsRequest& WithMaxResults(int value) {
    SetMaxResults(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The token for the next set of items to return. (You received this token from
   * a previous call.)</p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  ListTypeRegistrationsRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}
 private:
  RegistryType m_type{RegistryType::NOT_SET};
  bool m_typeHasBeenSet = false;

  Aws::String m_typeName;
  bool m_typeNameHasBeenSet = false;

  Aws::String m_typeArn;
  bool m_typeArnHasBeenSet = false;

  RegistrationStatus m_registrationStatusFilter{RegistrationStatus::NOT_SET};
  bool m_registrationStatusFilterHasBeenSet = false;

  int m_maxResults{0};
  bool m_maxResultsHasBeenSet = false;

  Aws::String m_nextToken;
  bool m_nextTokenHasBeenSet = false;
};

}  // namespace Model
}  // namespace CloudFormation
}  // namespace Aws
