﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/ecs/ECSRequest.h>
#include <aws/ecs/ECS_EXPORTS.h>
#include <aws/ecs/model/CapacityProviderStrategyItem.h>
#include <aws/ecs/model/LaunchType.h>
#include <aws/ecs/model/LoadBalancer.h>
#include <aws/ecs/model/NetworkConfiguration.h>
#include <aws/ecs/model/Scale.h>
#include <aws/ecs/model/ServiceRegistry.h>
#include <aws/ecs/model/Tag.h>

#include <utility>

namespace Aws {
namespace ECS {
namespace Model {

/**
 */
class CreateTaskSetRequest : public ECSRequest {
 public:
  AWS_ECS_API CreateTaskSetRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateTaskSet"; }

  AWS_ECS_API Aws::String SerializePayload() const override;

  AWS_ECS_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The short name or full Amazon Resource Name (ARN) of the service to create
   * the task set in.</p>
   */
  inline const Aws::String& GetService() const { return m_service; }
  inline bool ServiceHasBeenSet() const { return m_serviceHasBeenSet; }
  template <typename ServiceT = Aws::String>
  void SetService(ServiceT&& value) {
    m_serviceHasBeenSet = true;
    m_service = std::forward<ServiceT>(value);
  }
  template <typename ServiceT = Aws::String>
  CreateTaskSetRequest& WithService(ServiceT&& value) {
    SetService(std::forward<ServiceT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The short name or full Amazon Resource Name (ARN) of the cluster that hosts
   * the service to create the task set in.</p>
   */
  inline const Aws::String& GetCluster() const { return m_cluster; }
  inline bool ClusterHasBeenSet() const { return m_clusterHasBeenSet; }
  template <typename ClusterT = Aws::String>
  void SetCluster(ClusterT&& value) {
    m_clusterHasBeenSet = true;
    m_cluster = std::forward<ClusterT>(value);
  }
  template <typename ClusterT = Aws::String>
  CreateTaskSetRequest& WithCluster(ClusterT&& value) {
    SetCluster(std::forward<ClusterT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>An optional non-unique tag that identifies this task set in external systems.
   * If the task set is associated with a service discovery registry, the tasks in
   * this task set will have the <code>ECS_TASK_SET_EXTERNAL_ID</code> Cloud Map
   * attribute set to the provided value.</p>
   */
  inline const Aws::String& GetExternalId() const { return m_externalId; }
  inline bool ExternalIdHasBeenSet() const { return m_externalIdHasBeenSet; }
  template <typename ExternalIdT = Aws::String>
  void SetExternalId(ExternalIdT&& value) {
    m_externalIdHasBeenSet = true;
    m_externalId = std::forward<ExternalIdT>(value);
  }
  template <typename ExternalIdT = Aws::String>
  CreateTaskSetRequest& WithExternalId(ExternalIdT&& value) {
    SetExternalId(std::forward<ExternalIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The task definition for the tasks in the task set to use. If a revision isn't
   * specified, the latest <code>ACTIVE</code> revision is used.</p>
   */
  inline const Aws::String& GetTaskDefinition() const { return m_taskDefinition; }
  inline bool TaskDefinitionHasBeenSet() const { return m_taskDefinitionHasBeenSet; }
  template <typename TaskDefinitionT = Aws::String>
  void SetTaskDefinition(TaskDefinitionT&& value) {
    m_taskDefinitionHasBeenSet = true;
    m_taskDefinition = std::forward<TaskDefinitionT>(value);
  }
  template <typename TaskDefinitionT = Aws::String>
  CreateTaskSetRequest& WithTaskDefinition(TaskDefinitionT&& value) {
    SetTaskDefinition(std::forward<TaskDefinitionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>An object representing the network configuration for a task set.</p>
   */
  inline const NetworkConfiguration& GetNetworkConfiguration() const { return m_networkConfiguration; }
  inline bool NetworkConfigurationHasBeenSet() const { return m_networkConfigurationHasBeenSet; }
  template <typename NetworkConfigurationT = NetworkConfiguration>
  void SetNetworkConfiguration(NetworkConfigurationT&& value) {
    m_networkConfigurationHasBeenSet = true;
    m_networkConfiguration = std::forward<NetworkConfigurationT>(value);
  }
  template <typename NetworkConfigurationT = NetworkConfiguration>
  CreateTaskSetRequest& WithNetworkConfiguration(NetworkConfigurationT&& value) {
    SetNetworkConfiguration(std::forward<NetworkConfigurationT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A load balancer object representing the load balancer to use with the task
   * set. The supported load balancer types are either an Application Load Balancer
   * or a Network Load Balancer.</p>
   */
  inline const Aws::Vector<LoadBalancer>& GetLoadBalancers() const { return m_loadBalancers; }
  inline bool LoadBalancersHasBeenSet() const { return m_loadBalancersHasBeenSet; }
  template <typename LoadBalancersT = Aws::Vector<LoadBalancer>>
  void SetLoadBalancers(LoadBalancersT&& value) {
    m_loadBalancersHasBeenSet = true;
    m_loadBalancers = std::forward<LoadBalancersT>(value);
  }
  template <typename LoadBalancersT = Aws::Vector<LoadBalancer>>
  CreateTaskSetRequest& WithLoadBalancers(LoadBalancersT&& value) {
    SetLoadBalancers(std::forward<LoadBalancersT>(value));
    return *this;
  }
  template <typename LoadBalancersT = LoadBalancer>
  CreateTaskSetRequest& AddLoadBalancers(LoadBalancersT&& value) {
    m_loadBalancersHasBeenSet = true;
    m_loadBalancers.emplace_back(std::forward<LoadBalancersT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The details of the service discovery registries to assign to this task set.
   * For more information, see <a
   * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-discovery.html">Service
   * discovery</a>.</p>
   */
  inline const Aws::Vector<ServiceRegistry>& GetServiceRegistries() const { return m_serviceRegistries; }
  inline bool ServiceRegistriesHasBeenSet() const { return m_serviceRegistriesHasBeenSet; }
  template <typename ServiceRegistriesT = Aws::Vector<ServiceRegistry>>
  void SetServiceRegistries(ServiceRegistriesT&& value) {
    m_serviceRegistriesHasBeenSet = true;
    m_serviceRegistries = std::forward<ServiceRegistriesT>(value);
  }
  template <typename ServiceRegistriesT = Aws::Vector<ServiceRegistry>>
  CreateTaskSetRequest& WithServiceRegistries(ServiceRegistriesT&& value) {
    SetServiceRegistries(std::forward<ServiceRegistriesT>(value));
    return *this;
  }
  template <typename ServiceRegistriesT = ServiceRegistry>
  CreateTaskSetRequest& AddServiceRegistries(ServiceRegistriesT&& value) {
    m_serviceRegistriesHasBeenSet = true;
    m_serviceRegistries.emplace_back(std::forward<ServiceRegistriesT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The launch type that new tasks in the task set uses. For more information,
   * see <a
   * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/launch_types.html">Amazon
   * ECS launch types</a> in the <i>Amazon Elastic Container Service Developer
   * Guide</i>.</p> <p>If a <code>launchType</code> is specified, the
   * <code>capacityProviderStrategy</code> parameter must be omitted.</p>
   */
  inline LaunchType GetLaunchType() const { return m_launchType; }
  inline bool LaunchTypeHasBeenSet() const { return m_launchTypeHasBeenSet; }
  inline void SetLaunchType(LaunchType value) {
    m_launchTypeHasBeenSet = true;
    m_launchType = value;
  }
  inline CreateTaskSetRequest& WithLaunchType(LaunchType value) {
    SetLaunchType(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The capacity provider strategy to use for the task set.</p> <p>A capacity
   * provider strategy consists of one or more capacity providers along with the
   * <code>base</code> and <code>weight</code> to assign to them. A capacity provider
   * must be associated with the cluster to be used in a capacity provider strategy.
   * The <a
   * href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_PutClusterCapacityProviders.html">PutClusterCapacityProviders</a>
   * API is used to associate a capacity provider with a cluster. Only capacity
   * providers with an <code>ACTIVE</code> or <code>UPDATING</code> status can be
   * used.</p> <p>If a <code>capacityProviderStrategy</code> is specified, the
   * <code>launchType</code> parameter must be omitted. If no
   * <code>capacityProviderStrategy</code> or <code>launchType</code> is specified,
   * the <code>defaultCapacityProviderStrategy</code> for the cluster is used.</p>
   * <p>If specifying a capacity provider that uses an Auto Scaling group, the
   * capacity provider must already be created. New capacity providers can be created
   * with the <a
   * href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_CreateCapacityProviderProvider.html">CreateCapacityProviderProvider</a>API
   * operation.</p> <p>To use a Fargate capacity provider, specify either the
   * <code>FARGATE</code> or <code>FARGATE_SPOT</code> capacity providers. The
   * Fargate capacity providers are available to all accounts and only need to be
   * associated with a cluster to be used.</p> <p>The <a
   * href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_PutClusterCapacityProviders.html">PutClusterCapacityProviders</a>
   * API operation is used to update the list of available capacity providers for a
   * cluster after the cluster is created.</p>
   */
  inline const Aws::Vector<CapacityProviderStrategyItem>& GetCapacityProviderStrategy() const { return m_capacityProviderStrategy; }
  inline bool CapacityProviderStrategyHasBeenSet() const { return m_capacityProviderStrategyHasBeenSet; }
  template <typename CapacityProviderStrategyT = Aws::Vector<CapacityProviderStrategyItem>>
  void SetCapacityProviderStrategy(CapacityProviderStrategyT&& value) {
    m_capacityProviderStrategyHasBeenSet = true;
    m_capacityProviderStrategy = std::forward<CapacityProviderStrategyT>(value);
  }
  template <typename CapacityProviderStrategyT = Aws::Vector<CapacityProviderStrategyItem>>
  CreateTaskSetRequest& WithCapacityProviderStrategy(CapacityProviderStrategyT&& value) {
    SetCapacityProviderStrategy(std::forward<CapacityProviderStrategyT>(value));
    return *this;
  }
  template <typename CapacityProviderStrategyT = CapacityProviderStrategyItem>
  CreateTaskSetRequest& AddCapacityProviderStrategy(CapacityProviderStrategyT&& value) {
    m_capacityProviderStrategyHasBeenSet = true;
    m_capacityProviderStrategy.emplace_back(std::forward<CapacityProviderStrategyT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The platform version that the tasks in the task set uses. A platform version
   * is specified only for tasks using the Fargate launch type. If one isn't
   * specified, the <code>LATEST</code> platform version is used.</p>
   */
  inline const Aws::String& GetPlatformVersion() const { return m_platformVersion; }
  inline bool PlatformVersionHasBeenSet() const { return m_platformVersionHasBeenSet; }
  template <typename PlatformVersionT = Aws::String>
  void SetPlatformVersion(PlatformVersionT&& value) {
    m_platformVersionHasBeenSet = true;
    m_platformVersion = std::forward<PlatformVersionT>(value);
  }
  template <typename PlatformVersionT = Aws::String>
  CreateTaskSetRequest& WithPlatformVersion(PlatformVersionT&& value) {
    SetPlatformVersion(std::forward<PlatformVersionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A floating-point percentage of the desired number of tasks to place and keep
   * running in the task set.</p>
   */
  inline const Scale& GetScale() const { return m_scale; }
  inline bool ScaleHasBeenSet() const { return m_scaleHasBeenSet; }
  template <typename ScaleT = Scale>
  void SetScale(ScaleT&& value) {
    m_scaleHasBeenSet = true;
    m_scale = std::forward<ScaleT>(value);
  }
  template <typename ScaleT = Scale>
  CreateTaskSetRequest& WithScale(ScaleT&& value) {
    SetScale(std::forward<ScaleT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>An identifier that you provide to ensure the idempotency of the request. It
   * must be unique and is case sensitive. Up to 36 ASCII characters in the range of
   * 33-126 (inclusive) are allowed.</p>
   */
  inline const Aws::String& GetClientToken() const { return m_clientToken; }
  inline bool ClientTokenHasBeenSet() const { return m_clientTokenHasBeenSet; }
  template <typename ClientTokenT = Aws::String>
  void SetClientToken(ClientTokenT&& value) {
    m_clientTokenHasBeenSet = true;
    m_clientToken = std::forward<ClientTokenT>(value);
  }
  template <typename ClientTokenT = Aws::String>
  CreateTaskSetRequest& WithClientToken(ClientTokenT&& value) {
    SetClientToken(std::forward<ClientTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The metadata that you apply to the task set to help you categorize and
   * organize them. Each tag consists of a key and an optional value. You define
   * both. When a service is deleted, the tags are deleted.</p> <p>The following
   * basic restrictions apply to tags:</p> <ul> <li> <p>Maximum number of tags per
   * resource - 50</p> </li> <li> <p>For each resource, each tag key must be unique,
   * and each tag key can have only one value.</p> </li> <li> <p>Maximum key length -
   * 128 Unicode characters in UTF-8</p> </li> <li> <p>Maximum value length - 256
   * Unicode characters in UTF-8</p> </li> <li> <p>If your tagging schema is used
   * across multiple services and resources, remember that other services may have
   * restrictions on allowed characters. Generally allowed characters are: letters,
   * numbers, and spaces representable in UTF-8, and the following characters: + - =
   * . _ : / @.</p> </li> <li> <p>Tag keys and values are case-sensitive.</p> </li>
   * <li> <p>Do not use <code>aws:</code>, <code>AWS:</code>, or any upper or
   * lowercase combination of such as a prefix for either keys or values as it is
   * reserved for Amazon Web Services use. You cannot edit or delete tag keys or
   * values with this prefix. Tags with this prefix do not count against your tags
   * per resource limit.</p> </li> </ul>
   */
  inline const Aws::Vector<Tag>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Vector<Tag>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Vector<Tag>>
  CreateTaskSetRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsT = Tag>
  CreateTaskSetRequest& AddTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace_back(std::forward<TagsT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_service;
  bool m_serviceHasBeenSet = false;

  Aws::String m_cluster;
  bool m_clusterHasBeenSet = false;

  Aws::String m_externalId;
  bool m_externalIdHasBeenSet = false;

  Aws::String m_taskDefinition;
  bool m_taskDefinitionHasBeenSet = false;

  NetworkConfiguration m_networkConfiguration;
  bool m_networkConfigurationHasBeenSet = false;

  Aws::Vector<LoadBalancer> m_loadBalancers;
  bool m_loadBalancersHasBeenSet = false;

  Aws::Vector<ServiceRegistry> m_serviceRegistries;
  bool m_serviceRegistriesHasBeenSet = false;

  LaunchType m_launchType{LaunchType::NOT_SET};
  bool m_launchTypeHasBeenSet = false;

  Aws::Vector<CapacityProviderStrategyItem> m_capacityProviderStrategy;
  bool m_capacityProviderStrategyHasBeenSet = false;

  Aws::String m_platformVersion;
  bool m_platformVersionHasBeenSet = false;

  Scale m_scale;
  bool m_scaleHasBeenSet = false;

  Aws::String m_clientToken;
  bool m_clientTokenHasBeenSet = false;

  Aws::Vector<Tag> m_tags;
  bool m_tagsHasBeenSet = false;
};

}  // namespace Model
}  // namespace ECS
}  // namespace Aws
