﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/macie2/Macie2Request.h>
#include <aws/macie2/Macie2_EXPORTS.h>
#include <aws/macie2/model/BucketCriteriaAdditionalProperties.h>
#include <aws/macie2/model/BucketSortCriteria.h>

#include <utility>

namespace Aws {
namespace Macie2 {
namespace Model {

/**
 */
class DescribeBucketsRequest : public Macie2Request {
 public:
  AWS_MACIE2_API DescribeBucketsRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "DescribeBuckets"; }

  AWS_MACIE2_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The criteria to use to filter the query results.</p>
   */
  inline const Aws::Map<Aws::String, BucketCriteriaAdditionalProperties>& GetCriteria() const { return m_criteria; }
  inline bool CriteriaHasBeenSet() const { return m_criteriaHasBeenSet; }
  template <typename CriteriaT = Aws::Map<Aws::String, BucketCriteriaAdditionalProperties>>
  void SetCriteria(CriteriaT&& value) {
    m_criteriaHasBeenSet = true;
    m_criteria = std::forward<CriteriaT>(value);
  }
  template <typename CriteriaT = Aws::Map<Aws::String, BucketCriteriaAdditionalProperties>>
  DescribeBucketsRequest& WithCriteria(CriteriaT&& value) {
    SetCriteria(std::forward<CriteriaT>(value));
    return *this;
  }
  template <typename CriteriaKeyT = Aws::String, typename CriteriaValueT = BucketCriteriaAdditionalProperties>
  DescribeBucketsRequest& AddCriteria(CriteriaKeyT&& key, CriteriaValueT&& value) {
    m_criteriaHasBeenSet = true;
    m_criteria.emplace(std::forward<CriteriaKeyT>(key), std::forward<CriteriaValueT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of items to include in each page of the response. The
   * default value is 50.</p>
   */
  inline int GetMaxResults() const { return m_maxResults; }
  inline bool MaxResultsHasBeenSet() const { return m_maxResultsHasBeenSet; }
  inline void SetMaxResults(int value) {
    m_maxResultsHasBeenSet = true;
    m_maxResults = value;
  }
  inline DescribeBucketsRequest& WithMaxResults(int value) {
    SetMaxResults(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The nextToken string that specifies which page of results to return in a
   * paginated response.</p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  DescribeBucketsRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The criteria to use to sort the query results.</p>
   */
  inline const BucketSortCriteria& GetSortCriteria() const { return m_sortCriteria; }
  inline bool SortCriteriaHasBeenSet() const { return m_sortCriteriaHasBeenSet; }
  template <typename SortCriteriaT = BucketSortCriteria>
  void SetSortCriteria(SortCriteriaT&& value) {
    m_sortCriteriaHasBeenSet = true;
    m_sortCriteria = std::forward<SortCriteriaT>(value);
  }
  template <typename SortCriteriaT = BucketSortCriteria>
  DescribeBucketsRequest& WithSortCriteria(SortCriteriaT&& value) {
    SetSortCriteria(std::forward<SortCriteriaT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::Map<Aws::String, BucketCriteriaAdditionalProperties> m_criteria;
  bool m_criteriaHasBeenSet = false;

  int m_maxResults{0};
  bool m_maxResultsHasBeenSet = false;

  Aws::String m_nextToken;
  bool m_nextTokenHasBeenSet = false;

  BucketSortCriteria m_sortCriteria;
  bool m_sortCriteriaHasBeenSet = false;
};

}  // namespace Model
}  // namespace Macie2
}  // namespace Aws
