﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/sso-admin/SSOAdminRequest.h>
#include <aws/sso-admin/SSOAdmin_EXPORTS.h>
#include <aws/sso-admin/model/ProvisionTargetType.h>

#include <utility>

namespace Aws {
namespace SSOAdmin {
namespace Model {

/**
 */
class ProvisionPermissionSetRequest : public SSOAdminRequest {
 public:
  AWS_SSOADMIN_API ProvisionPermissionSetRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ProvisionPermissionSet"; }

  AWS_SSOADMIN_API Aws::String SerializePayload() const override;

  AWS_SSOADMIN_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The ARN of the IAM Identity Center instance under which the operation will be
   * executed. For more information about ARNs, see <a
   * href="/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource Names
   * (ARNs) and Amazon Web Services Service Namespaces</a> in the <i>Amazon Web
   * Services General Reference</i>.</p>
   */
  inline const Aws::String& GetInstanceArn() const { return m_instanceArn; }
  inline bool InstanceArnHasBeenSet() const { return m_instanceArnHasBeenSet; }
  template <typename InstanceArnT = Aws::String>
  void SetInstanceArn(InstanceArnT&& value) {
    m_instanceArnHasBeenSet = true;
    m_instanceArn = std::forward<InstanceArnT>(value);
  }
  template <typename InstanceArnT = Aws::String>
  ProvisionPermissionSetRequest& WithInstanceArn(InstanceArnT&& value) {
    SetInstanceArn(std::forward<InstanceArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ARN of the permission set.</p>
   */
  inline const Aws::String& GetPermissionSetArn() const { return m_permissionSetArn; }
  inline bool PermissionSetArnHasBeenSet() const { return m_permissionSetArnHasBeenSet; }
  template <typename PermissionSetArnT = Aws::String>
  void SetPermissionSetArn(PermissionSetArnT&& value) {
    m_permissionSetArnHasBeenSet = true;
    m_permissionSetArn = std::forward<PermissionSetArnT>(value);
  }
  template <typename PermissionSetArnT = Aws::String>
  ProvisionPermissionSetRequest& WithPermissionSetArn(PermissionSetArnT&& value) {
    SetPermissionSetArn(std::forward<PermissionSetArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>TargetID is an Amazon Web Services account identifier, (For example,
   * 123456789012).</p>
   */
  inline const Aws::String& GetTargetId() const { return m_targetId; }
  inline bool TargetIdHasBeenSet() const { return m_targetIdHasBeenSet; }
  template <typename TargetIdT = Aws::String>
  void SetTargetId(TargetIdT&& value) {
    m_targetIdHasBeenSet = true;
    m_targetId = std::forward<TargetIdT>(value);
  }
  template <typename TargetIdT = Aws::String>
  ProvisionPermissionSetRequest& WithTargetId(TargetIdT&& value) {
    SetTargetId(std::forward<TargetIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The entity type for which the assignment will be created.</p>
   */
  inline ProvisionTargetType GetTargetType() const { return m_targetType; }
  inline bool TargetTypeHasBeenSet() const { return m_targetTypeHasBeenSet; }
  inline void SetTargetType(ProvisionTargetType value) {
    m_targetTypeHasBeenSet = true;
    m_targetType = value;
  }
  inline ProvisionPermissionSetRequest& WithTargetType(ProvisionTargetType value) {
    SetTargetType(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_instanceArn;
  bool m_instanceArnHasBeenSet = false;

  Aws::String m_permissionSetArn;
  bool m_permissionSetArnHasBeenSet = false;

  Aws::String m_targetId;
  bool m_targetIdHasBeenSet = false;

  ProvisionTargetType m_targetType{ProvisionTargetType::NOT_SET};
  bool m_targetTypeHasBeenSet = false;
};

}  // namespace Model
}  // namespace SSOAdmin
}  // namespace Aws
