﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/datazone/DataZoneRequest.h>
#include <aws/datazone/DataZone_EXPORTS.h>
#include <aws/datazone/model/DataSourceStatus.h>

#include <utility>

namespace Aws {
namespace Http {
class URI;
}  // namespace Http
namespace DataZone {
namespace Model {

/**
 */
class ListDataSourcesRequest : public DataZoneRequest {
 public:
  AWS_DATAZONE_API ListDataSourcesRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ListDataSources"; }

  AWS_DATAZONE_API Aws::String SerializePayload() const override;

  AWS_DATAZONE_API void AddQueryStringParameters(Aws::Http::URI& uri) const override;

  ///@{
  /**
   * <p>The ID of the connection.</p>
   */
  inline const Aws::String& GetConnectionIdentifier() const { return m_connectionIdentifier; }
  inline bool ConnectionIdentifierHasBeenSet() const { return m_connectionIdentifierHasBeenSet; }
  template <typename ConnectionIdentifierT = Aws::String>
  void SetConnectionIdentifier(ConnectionIdentifierT&& value) {
    m_connectionIdentifierHasBeenSet = true;
    m_connectionIdentifier = std::forward<ConnectionIdentifierT>(value);
  }
  template <typename ConnectionIdentifierT = Aws::String>
  ListDataSourcesRequest& WithConnectionIdentifier(ConnectionIdentifierT&& value) {
    SetConnectionIdentifier(std::forward<ConnectionIdentifierT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The identifier of the Amazon DataZone domain in which to list the data
   * sources.</p>
   */
  inline const Aws::String& GetDomainIdentifier() const { return m_domainIdentifier; }
  inline bool DomainIdentifierHasBeenSet() const { return m_domainIdentifierHasBeenSet; }
  template <typename DomainIdentifierT = Aws::String>
  void SetDomainIdentifier(DomainIdentifierT&& value) {
    m_domainIdentifierHasBeenSet = true;
    m_domainIdentifier = std::forward<DomainIdentifierT>(value);
  }
  template <typename DomainIdentifierT = Aws::String>
  ListDataSourcesRequest& WithDomainIdentifier(DomainIdentifierT&& value) {
    SetDomainIdentifier(std::forward<DomainIdentifierT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The identifier of the environment in which to list the data sources.</p>
   */
  inline const Aws::String& GetEnvironmentIdentifier() const { return m_environmentIdentifier; }
  inline bool EnvironmentIdentifierHasBeenSet() const { return m_environmentIdentifierHasBeenSet; }
  template <typename EnvironmentIdentifierT = Aws::String>
  void SetEnvironmentIdentifier(EnvironmentIdentifierT&& value) {
    m_environmentIdentifierHasBeenSet = true;
    m_environmentIdentifier = std::forward<EnvironmentIdentifierT>(value);
  }
  template <typename EnvironmentIdentifierT = Aws::String>
  ListDataSourcesRequest& WithEnvironmentIdentifier(EnvironmentIdentifierT&& value) {
    SetEnvironmentIdentifier(std::forward<EnvironmentIdentifierT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of data sources to return in a single call to
   * <code>ListDataSources</code>. When the number of data sources to be listed is
   * greater than the value of <code>MaxResults</code>, the response contains a
   * <code>NextToken</code> value that you can use in a subsequent call to
   * <code>ListDataSources</code> to list the next set of data sources.</p>
   */
  inline int GetMaxResults() const { return m_maxResults; }
  inline bool MaxResultsHasBeenSet() const { return m_maxResultsHasBeenSet; }
  inline void SetMaxResults(int value) {
    m_maxResultsHasBeenSet = true;
    m_maxResults = value;
  }
  inline ListDataSourcesRequest& WithMaxResults(int value) {
    SetMaxResults(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the data source.</p>
   */
  inline const Aws::String& GetName() const { return m_name; }
  inline bool NameHasBeenSet() const { return m_nameHasBeenSet; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  ListDataSourcesRequest& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>When the number of data sources is greater than the default value for the
   * <code>MaxResults</code> parameter, or if you explicitly specify a value for
   * <code>MaxResults</code> that is less than the number of data sources, the
   * response includes a pagination token named <code>NextToken</code>. You can
   * specify this <code>NextToken</code> value in a subsequent call to
   * <code>ListDataSources</code> to list the next set of data sources.</p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  ListDataSourcesRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The identifier of the project in which to list data sources.</p>
   */
  inline const Aws::String& GetProjectIdentifier() const { return m_projectIdentifier; }
  inline bool ProjectIdentifierHasBeenSet() const { return m_projectIdentifierHasBeenSet; }
  template <typename ProjectIdentifierT = Aws::String>
  void SetProjectIdentifier(ProjectIdentifierT&& value) {
    m_projectIdentifierHasBeenSet = true;
    m_projectIdentifier = std::forward<ProjectIdentifierT>(value);
  }
  template <typename ProjectIdentifierT = Aws::String>
  ListDataSourcesRequest& WithProjectIdentifier(ProjectIdentifierT&& value) {
    SetProjectIdentifier(std::forward<ProjectIdentifierT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The status of the data source.</p>
   */
  inline DataSourceStatus GetStatus() const { return m_status; }
  inline bool StatusHasBeenSet() const { return m_statusHasBeenSet; }
  inline void SetStatus(DataSourceStatus value) {
    m_statusHasBeenSet = true;
    m_status = value;
  }
  inline ListDataSourcesRequest& WithStatus(DataSourceStatus value) {
    SetStatus(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The type of the data source.</p>
   */
  inline const Aws::String& GetType() const { return m_type; }
  inline bool TypeHasBeenSet() const { return m_typeHasBeenSet; }
  template <typename TypeT = Aws::String>
  void SetType(TypeT&& value) {
    m_typeHasBeenSet = true;
    m_type = std::forward<TypeT>(value);
  }
  template <typename TypeT = Aws::String>
  ListDataSourcesRequest& WithType(TypeT&& value) {
    SetType(std::forward<TypeT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_connectionIdentifier;
  bool m_connectionIdentifierHasBeenSet = false;

  Aws::String m_domainIdentifier;
  bool m_domainIdentifierHasBeenSet = false;

  Aws::String m_environmentIdentifier;
  bool m_environmentIdentifierHasBeenSet = false;

  int m_maxResults{0};
  bool m_maxResultsHasBeenSet = false;

  Aws::String m_name;
  bool m_nameHasBeenSet = false;

  Aws::String m_nextToken;
  bool m_nextTokenHasBeenSet = false;

  Aws::String m_projectIdentifier;
  bool m_projectIdentifierHasBeenSet = false;

  DataSourceStatus m_status{DataSourceStatus::NOT_SET};
  bool m_statusHasBeenSet = false;

  Aws::String m_type;
  bool m_typeHasBeenSet = false;
};

}  // namespace Model
}  // namespace DataZone
}  // namespace Aws
