﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/UUID.h>
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/imagebuilder/ImagebuilderRequest.h>
#include <aws/imagebuilder/Imagebuilder_EXPORTS.h>
#include <aws/imagebuilder/model/ImageLoggingConfiguration.h>
#include <aws/imagebuilder/model/Platform.h>

#include <utility>

namespace Aws {
namespace imagebuilder {
namespace Model {

/**
 */
class ImportVmImageRequest : public ImagebuilderRequest {
 public:
  AWS_IMAGEBUILDER_API ImportVmImageRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ImportVmImage"; }

  AWS_IMAGEBUILDER_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The name of the base image that is created by the import process.</p>
   */
  inline const Aws::String& GetName() const { return m_name; }
  inline bool NameHasBeenSet() const { return m_nameHasBeenSet; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  ImportVmImageRequest& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The semantic version to attach to the base image that was created during the
   * import process. This version follows the semantic version syntax.</p>
   * <p>The semantic version has four nodes:
   * &lt;major&gt;.&lt;minor&gt;.&lt;patch&gt;/&lt;build&gt;. You can assign values
   * for the first three, and can filter on all of them.</p> <p> <b>Assignment:</b>
   * For the first three nodes you can assign any positive integer value, including
   * zero, with an upper limit of 2^30-1, or 1073741823 for each node. Image Builder
   * automatically assigns the build number to the fourth node.</p> <p>
   * <b>Patterns:</b> You can use any numeric pattern that adheres to the assignment
   * requirements for the nodes that you can assign. For example, you might choose a
   * software version pattern, such as 1.0.0, or a date, such as 2021.01.01.</p>
   *
   */
  inline const Aws::String& GetSemanticVersion() const { return m_semanticVersion; }
  inline bool SemanticVersionHasBeenSet() const { return m_semanticVersionHasBeenSet; }
  template <typename SemanticVersionT = Aws::String>
  void SetSemanticVersion(SemanticVersionT&& value) {
    m_semanticVersionHasBeenSet = true;
    m_semanticVersion = std::forward<SemanticVersionT>(value);
  }
  template <typename SemanticVersionT = Aws::String>
  ImportVmImageRequest& WithSemanticVersion(SemanticVersionT&& value) {
    SetSemanticVersion(std::forward<SemanticVersionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The description for the base image that is created by the import process.</p>
   */
  inline const Aws::String& GetDescription() const { return m_description; }
  inline bool DescriptionHasBeenSet() const { return m_descriptionHasBeenSet; }
  template <typename DescriptionT = Aws::String>
  void SetDescription(DescriptionT&& value) {
    m_descriptionHasBeenSet = true;
    m_description = std::forward<DescriptionT>(value);
  }
  template <typename DescriptionT = Aws::String>
  ImportVmImageRequest& WithDescription(DescriptionT&& value) {
    SetDescription(std::forward<DescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The operating system platform for the imported VM.</p>
   */
  inline Platform GetPlatform() const { return m_platform; }
  inline bool PlatformHasBeenSet() const { return m_platformHasBeenSet; }
  inline void SetPlatform(Platform value) {
    m_platformHasBeenSet = true;
    m_platform = value;
  }
  inline ImportVmImageRequest& WithPlatform(Platform value) {
    SetPlatform(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The operating system version for the imported VM.</p>
   */
  inline const Aws::String& GetOsVersion() const { return m_osVersion; }
  inline bool OsVersionHasBeenSet() const { return m_osVersionHasBeenSet; }
  template <typename OsVersionT = Aws::String>
  void SetOsVersion(OsVersionT&& value) {
    m_osVersionHasBeenSet = true;
    m_osVersion = std::forward<OsVersionT>(value);
  }
  template <typename OsVersionT = Aws::String>
  ImportVmImageRequest& WithOsVersion(OsVersionT&& value) {
    SetOsVersion(std::forward<OsVersionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The <code>importTaskId</code> (API) or <code>ImportTaskId</code> (CLI) from
   * the Amazon EC2 VM import process. Image Builder retrieves information from the
   * import process to pull in the AMI that is created from the VM source as the base
   * image for your recipe.</p>
   */
  inline const Aws::String& GetVmImportTaskId() const { return m_vmImportTaskId; }
  inline bool VmImportTaskIdHasBeenSet() const { return m_vmImportTaskIdHasBeenSet; }
  template <typename VmImportTaskIdT = Aws::String>
  void SetVmImportTaskId(VmImportTaskIdT&& value) {
    m_vmImportTaskIdHasBeenSet = true;
    m_vmImportTaskId = std::forward<VmImportTaskIdT>(value);
  }
  template <typename VmImportTaskIdT = Aws::String>
  ImportVmImageRequest& WithVmImportTaskId(VmImportTaskIdT&& value) {
    SetVmImportTaskId(std::forward<VmImportTaskIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Define logging configuration for the image build process.</p>
   */
  inline const ImageLoggingConfiguration& GetLoggingConfiguration() const { return m_loggingConfiguration; }
  inline bool LoggingConfigurationHasBeenSet() const { return m_loggingConfigurationHasBeenSet; }
  template <typename LoggingConfigurationT = ImageLoggingConfiguration>
  void SetLoggingConfiguration(LoggingConfigurationT&& value) {
    m_loggingConfigurationHasBeenSet = true;
    m_loggingConfiguration = std::forward<LoggingConfigurationT>(value);
  }
  template <typename LoggingConfigurationT = ImageLoggingConfiguration>
  ImportVmImageRequest& WithLoggingConfiguration(LoggingConfigurationT&& value) {
    SetLoggingConfiguration(std::forward<LoggingConfigurationT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Tags that are attached to the import resources.</p>
   */
  inline const Aws::Map<Aws::String, Aws::String>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  ImportVmImageRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsKeyT = Aws::String, typename TagsValueT = Aws::String>
  ImportVmImageRequest& AddTags(TagsKeyT&& key, TagsValueT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace(std::forward<TagsKeyT>(key), std::forward<TagsValueT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Unique, case-sensitive identifier you provide to ensure idempotency of the
   * request. For more information, see <a
   * href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/Run_Instance_Idempotency.html">Ensuring
   * idempotency</a> in the <i>Amazon EC2 API Reference</i>.</p>
   */
  inline const Aws::String& GetClientToken() const { return m_clientToken; }
  inline bool ClientTokenHasBeenSet() const { return m_clientTokenHasBeenSet; }
  template <typename ClientTokenT = Aws::String>
  void SetClientToken(ClientTokenT&& value) {
    m_clientTokenHasBeenSet = true;
    m_clientToken = std::forward<ClientTokenT>(value);
  }
  template <typename ClientTokenT = Aws::String>
  ImportVmImageRequest& WithClientToken(ClientTokenT&& value) {
    SetClientToken(std::forward<ClientTokenT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_name;
  bool m_nameHasBeenSet = false;

  Aws::String m_semanticVersion;
  bool m_semanticVersionHasBeenSet = false;

  Aws::String m_description;
  bool m_descriptionHasBeenSet = false;

  Platform m_platform{Platform::NOT_SET};
  bool m_platformHasBeenSet = false;

  Aws::String m_osVersion;
  bool m_osVersionHasBeenSet = false;

  Aws::String m_vmImportTaskId;
  bool m_vmImportTaskIdHasBeenSet = false;

  ImageLoggingConfiguration m_loggingConfiguration;
  bool m_loggingConfigurationHasBeenSet = false;

  Aws::Map<Aws::String, Aws::String> m_tags;
  bool m_tagsHasBeenSet = false;

  Aws::String m_clientToken{Aws::Utils::UUID::PseudoRandomUUID()};
  bool m_clientTokenHasBeenSet = true;
};

}  // namespace Model
}  // namespace imagebuilder
}  // namespace Aws
