﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/iotwireless/IoTWirelessRequest.h>
#include <aws/iotwireless/IoTWireless_EXPORTS.h>
#include <aws/iotwireless/model/WirelessDeviceType.h>

#include <utility>

namespace Aws {
namespace Http {
class URI;
}  // namespace Http
namespace IoTWireless {
namespace Model {

/**
 */
class DeleteQueuedMessagesRequest : public IoTWirelessRequest {
 public:
  AWS_IOTWIRELESS_API DeleteQueuedMessagesRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "DeleteQueuedMessages"; }

  AWS_IOTWIRELESS_API Aws::String SerializePayload() const override;

  AWS_IOTWIRELESS_API void AddQueryStringParameters(Aws::Http::URI& uri) const override;

  ///@{
  /**
   * <p>The ID of a given wireless device for which downlink messages will be
   * deleted.</p>
   */
  inline const Aws::String& GetId() const { return m_id; }
  inline bool IdHasBeenSet() const { return m_idHasBeenSet; }
  template <typename IdT = Aws::String>
  void SetId(IdT&& value) {
    m_idHasBeenSet = true;
    m_id = std::forward<IdT>(value);
  }
  template <typename IdT = Aws::String>
  DeleteQueuedMessagesRequest& WithId(IdT&& value) {
    SetId(std::forward<IdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>If message ID is <code>"*"</code>, it cleares the entire downlink queue for a
   * given device, specified by the wireless device ID. Otherwise, the downlink
   * message with the specified message ID will be deleted.</p>
   */
  inline const Aws::String& GetMessageId() const { return m_messageId; }
  inline bool MessageIdHasBeenSet() const { return m_messageIdHasBeenSet; }
  template <typename MessageIdT = Aws::String>
  void SetMessageId(MessageIdT&& value) {
    m_messageIdHasBeenSet = true;
    m_messageId = std::forward<MessageIdT>(value);
  }
  template <typename MessageIdT = Aws::String>
  DeleteQueuedMessagesRequest& WithMessageId(MessageIdT&& value) {
    SetMessageId(std::forward<MessageIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The wireless device type, which can be either Sidewalk or LoRaWAN.</p>
   */
  inline WirelessDeviceType GetWirelessDeviceType() const { return m_wirelessDeviceType; }
  inline bool WirelessDeviceTypeHasBeenSet() const { return m_wirelessDeviceTypeHasBeenSet; }
  inline void SetWirelessDeviceType(WirelessDeviceType value) {
    m_wirelessDeviceTypeHasBeenSet = true;
    m_wirelessDeviceType = value;
  }
  inline DeleteQueuedMessagesRequest& WithWirelessDeviceType(WirelessDeviceType value) {
    SetWirelessDeviceType(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_id;
  bool m_idHasBeenSet = false;

  Aws::String m_messageId;
  bool m_messageIdHasBeenSet = false;

  WirelessDeviceType m_wirelessDeviceType{WirelessDeviceType::NOT_SET};
  bool m_wirelessDeviceTypeHasBeenSet = false;
};

}  // namespace Model
}  // namespace IoTWireless
}  // namespace Aws
