﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/DateTime.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/macie2/Macie2_EXPORTS.h>
#include <aws/macie2/model/KeyValuePair.h>
#include <aws/macie2/model/ServerSideEncryption.h>
#include <aws/macie2/model/StorageClass.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace Macie2 {
namespace Model {

/**
 * <p>Provides information about the S3 object that a finding applies
 * to.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/macie2-2020-01-01/S3Object">AWS API
 * Reference</a></p>
 */
class S3Object {
 public:
  AWS_MACIE2_API S3Object() = default;
  AWS_MACIE2_API S3Object(Aws::Utils::Json::JsonView jsonValue);
  AWS_MACIE2_API S3Object& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_MACIE2_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the bucket that contains the object.</p>
   */
  inline const Aws::String& GetBucketArn() const { return m_bucketArn; }
  inline bool BucketArnHasBeenSet() const { return m_bucketArnHasBeenSet; }
  template <typename BucketArnT = Aws::String>
  void SetBucketArn(BucketArnT&& value) {
    m_bucketArnHasBeenSet = true;
    m_bucketArn = std::forward<BucketArnT>(value);
  }
  template <typename BucketArnT = Aws::String>
  S3Object& WithBucketArn(BucketArnT&& value) {
    SetBucketArn(std::forward<BucketArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The entity tag (ETag) that identifies the affected version of the object. If
   * the object was overwritten or changed after Amazon Macie produced the finding,
   * this value might be different from the current ETag for the object.</p>
   */
  inline const Aws::String& GetETag() const { return m_eTag; }
  inline bool ETagHasBeenSet() const { return m_eTagHasBeenSet; }
  template <typename ETagT = Aws::String>
  void SetETag(ETagT&& value) {
    m_eTagHasBeenSet = true;
    m_eTag = std::forward<ETagT>(value);
  }
  template <typename ETagT = Aws::String>
  S3Object& WithETag(ETagT&& value) {
    SetETag(std::forward<ETagT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The file name extension of the object. If the object doesn't have a file name
   * extension, this value is "".</p>
   */
  inline const Aws::String& GetExtension() const { return m_extension; }
  inline bool ExtensionHasBeenSet() const { return m_extensionHasBeenSet; }
  template <typename ExtensionT = Aws::String>
  void SetExtension(ExtensionT&& value) {
    m_extensionHasBeenSet = true;
    m_extension = std::forward<ExtensionT>(value);
  }
  template <typename ExtensionT = Aws::String>
  S3Object& WithExtension(ExtensionT&& value) {
    SetExtension(std::forward<ExtensionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The full name (<i>key</i>) of the object, including the object's prefix if
   * applicable.</p>
   */
  inline const Aws::String& GetKey() const { return m_key; }
  inline bool KeyHasBeenSet() const { return m_keyHasBeenSet; }
  template <typename KeyT = Aws::String>
  void SetKey(KeyT&& value) {
    m_keyHasBeenSet = true;
    m_key = std::forward<KeyT>(value);
  }
  template <typename KeyT = Aws::String>
  S3Object& WithKey(KeyT&& value) {
    SetKey(std::forward<KeyT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The date and time, in UTC and extended ISO 8601 format, when the object was
   * last modified.</p>
   */
  inline const Aws::Utils::DateTime& GetLastModified() const { return m_lastModified; }
  inline bool LastModifiedHasBeenSet() const { return m_lastModifiedHasBeenSet; }
  template <typename LastModifiedT = Aws::Utils::DateTime>
  void SetLastModified(LastModifiedT&& value) {
    m_lastModifiedHasBeenSet = true;
    m_lastModified = std::forward<LastModifiedT>(value);
  }
  template <typename LastModifiedT = Aws::Utils::DateTime>
  S3Object& WithLastModified(LastModifiedT&& value) {
    SetLastModified(std::forward<LastModifiedT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The full path to the affected object, including the name of the affected
   * bucket and the object's name (key).</p>
   */
  inline const Aws::String& GetPath() const { return m_path; }
  inline bool PathHasBeenSet() const { return m_pathHasBeenSet; }
  template <typename PathT = Aws::String>
  void SetPath(PathT&& value) {
    m_pathHasBeenSet = true;
    m_path = std::forward<PathT>(value);
  }
  template <typename PathT = Aws::String>
  S3Object& WithPath(PathT&& value) {
    SetPath(std::forward<PathT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies whether the object is publicly accessible due to the combination of
   * permissions settings that apply to the object.</p>
   */
  inline bool GetPublicAccess() const { return m_publicAccess; }
  inline bool PublicAccessHasBeenSet() const { return m_publicAccessHasBeenSet; }
  inline void SetPublicAccess(bool value) {
    m_publicAccessHasBeenSet = true;
    m_publicAccess = value;
  }
  inline S3Object& WithPublicAccess(bool value) {
    SetPublicAccess(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The type of server-side encryption that was used to encrypt the object.</p>
   */
  inline const ServerSideEncryption& GetServerSideEncryption() const { return m_serverSideEncryption; }
  inline bool ServerSideEncryptionHasBeenSet() const { return m_serverSideEncryptionHasBeenSet; }
  template <typename ServerSideEncryptionT = ServerSideEncryption>
  void SetServerSideEncryption(ServerSideEncryptionT&& value) {
    m_serverSideEncryptionHasBeenSet = true;
    m_serverSideEncryption = std::forward<ServerSideEncryptionT>(value);
  }
  template <typename ServerSideEncryptionT = ServerSideEncryption>
  S3Object& WithServerSideEncryption(ServerSideEncryptionT&& value) {
    SetServerSideEncryption(std::forward<ServerSideEncryptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The total storage size, in bytes, of the object.</p>
   */
  inline long long GetSize() const { return m_size; }
  inline bool SizeHasBeenSet() const { return m_sizeHasBeenSet; }
  inline void SetSize(long long value) {
    m_sizeHasBeenSet = true;
    m_size = value;
  }
  inline S3Object& WithSize(long long value) {
    SetSize(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The storage class of the object.</p>
   */
  inline StorageClass GetStorageClass() const { return m_storageClass; }
  inline bool StorageClassHasBeenSet() const { return m_storageClassHasBeenSet; }
  inline void SetStorageClass(StorageClass value) {
    m_storageClassHasBeenSet = true;
    m_storageClass = value;
  }
  inline S3Object& WithStorageClass(StorageClass value) {
    SetStorageClass(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The tags that are associated with the object.</p>
   */
  inline const Aws::Vector<KeyValuePair>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Vector<KeyValuePair>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Vector<KeyValuePair>>
  S3Object& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsT = KeyValuePair>
  S3Object& AddTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace_back(std::forward<TagsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The identifier for the affected version of the object.</p>
   */
  inline const Aws::String& GetVersionId() const { return m_versionId; }
  inline bool VersionIdHasBeenSet() const { return m_versionIdHasBeenSet; }
  template <typename VersionIdT = Aws::String>
  void SetVersionId(VersionIdT&& value) {
    m_versionIdHasBeenSet = true;
    m_versionId = std::forward<VersionIdT>(value);
  }
  template <typename VersionIdT = Aws::String>
  S3Object& WithVersionId(VersionIdT&& value) {
    SetVersionId(std::forward<VersionIdT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_bucketArn;
  bool m_bucketArnHasBeenSet = false;

  Aws::String m_eTag;
  bool m_eTagHasBeenSet = false;

  Aws::String m_extension;
  bool m_extensionHasBeenSet = false;

  Aws::String m_key;
  bool m_keyHasBeenSet = false;

  Aws::Utils::DateTime m_lastModified{};
  bool m_lastModifiedHasBeenSet = false;

  Aws::String m_path;
  bool m_pathHasBeenSet = false;

  bool m_publicAccess{false};
  bool m_publicAccessHasBeenSet = false;

  ServerSideEncryption m_serverSideEncryption;
  bool m_serverSideEncryptionHasBeenSet = false;

  long long m_size{0};
  bool m_sizeHasBeenSet = false;

  StorageClass m_storageClass{StorageClass::NOT_SET};
  bool m_storageClassHasBeenSet = false;

  Aws::Vector<KeyValuePair> m_tags;
  bool m_tagsHasBeenSet = false;

  Aws::String m_versionId;
  bool m_versionIdHasBeenSet = false;
};

}  // namespace Model
}  // namespace Macie2
}  // namespace Aws
