﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/pipes/Pipes_EXPORTS.h>
#include <aws/pipes/model/BatchEnvironmentVariable.h>
#include <aws/pipes/model/BatchResourceRequirement.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace Pipes {
namespace Model {

/**
 * <p>The overrides that are sent to a container.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/pipes-2015-10-07/BatchContainerOverrides">AWS
 * API Reference</a></p>
 */
class BatchContainerOverrides {
 public:
  AWS_PIPES_API BatchContainerOverrides() = default;
  AWS_PIPES_API BatchContainerOverrides(Aws::Utils::Json::JsonView jsonValue);
  AWS_PIPES_API BatchContainerOverrides& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_PIPES_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>The command to send to the container that overrides the default command from
   * the Docker image or the task definition.</p>
   */
  inline const Aws::Vector<Aws::String>& GetCommand() const { return m_command; }
  inline bool CommandHasBeenSet() const { return m_commandHasBeenSet; }
  template <typename CommandT = Aws::Vector<Aws::String>>
  void SetCommand(CommandT&& value) {
    m_commandHasBeenSet = true;
    m_command = std::forward<CommandT>(value);
  }
  template <typename CommandT = Aws::Vector<Aws::String>>
  BatchContainerOverrides& WithCommand(CommandT&& value) {
    SetCommand(std::forward<CommandT>(value));
    return *this;
  }
  template <typename CommandT = Aws::String>
  BatchContainerOverrides& AddCommand(CommandT&& value) {
    m_commandHasBeenSet = true;
    m_command.emplace_back(std::forward<CommandT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The environment variables to send to the container. You can add new
   * environment variables, which are added to the container at launch, or you can
   * override the existing environment variables from the Docker image or the task
   * definition.</p>  <p>Environment variables cannot start with
   * "<code>Batch</code>". This naming convention is reserved for variables that
   * Batch sets.</p>
   */
  inline const Aws::Vector<BatchEnvironmentVariable>& GetEnvironment() const { return m_environment; }
  inline bool EnvironmentHasBeenSet() const { return m_environmentHasBeenSet; }
  template <typename EnvironmentT = Aws::Vector<BatchEnvironmentVariable>>
  void SetEnvironment(EnvironmentT&& value) {
    m_environmentHasBeenSet = true;
    m_environment = std::forward<EnvironmentT>(value);
  }
  template <typename EnvironmentT = Aws::Vector<BatchEnvironmentVariable>>
  BatchContainerOverrides& WithEnvironment(EnvironmentT&& value) {
    SetEnvironment(std::forward<EnvironmentT>(value));
    return *this;
  }
  template <typename EnvironmentT = BatchEnvironmentVariable>
  BatchContainerOverrides& AddEnvironment(EnvironmentT&& value) {
    m_environmentHasBeenSet = true;
    m_environment.emplace_back(std::forward<EnvironmentT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The instance type to use for a multi-node parallel job.</p>  <p>This
   * parameter isn't applicable to single-node container jobs or jobs that run on
   * Fargate resources, and shouldn't be provided.</p>
   */
  inline const Aws::String& GetInstanceType() const { return m_instanceType; }
  inline bool InstanceTypeHasBeenSet() const { return m_instanceTypeHasBeenSet; }
  template <typename InstanceTypeT = Aws::String>
  void SetInstanceType(InstanceTypeT&& value) {
    m_instanceTypeHasBeenSet = true;
    m_instanceType = std::forward<InstanceTypeT>(value);
  }
  template <typename InstanceTypeT = Aws::String>
  BatchContainerOverrides& WithInstanceType(InstanceTypeT&& value) {
    SetInstanceType(std::forward<InstanceTypeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The type and amount of resources to assign to a container. This overrides the
   * settings in the job definition. The supported resources include
   * <code>GPU</code>, <code>MEMORY</code>, and <code>VCPU</code>.</p>
   */
  inline const Aws::Vector<BatchResourceRequirement>& GetResourceRequirements() const { return m_resourceRequirements; }
  inline bool ResourceRequirementsHasBeenSet() const { return m_resourceRequirementsHasBeenSet; }
  template <typename ResourceRequirementsT = Aws::Vector<BatchResourceRequirement>>
  void SetResourceRequirements(ResourceRequirementsT&& value) {
    m_resourceRequirementsHasBeenSet = true;
    m_resourceRequirements = std::forward<ResourceRequirementsT>(value);
  }
  template <typename ResourceRequirementsT = Aws::Vector<BatchResourceRequirement>>
  BatchContainerOverrides& WithResourceRequirements(ResourceRequirementsT&& value) {
    SetResourceRequirements(std::forward<ResourceRequirementsT>(value));
    return *this;
  }
  template <typename ResourceRequirementsT = BatchResourceRequirement>
  BatchContainerOverrides& AddResourceRequirements(ResourceRequirementsT&& value) {
    m_resourceRequirementsHasBeenSet = true;
    m_resourceRequirements.emplace_back(std::forward<ResourceRequirementsT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::Vector<Aws::String> m_command;
  bool m_commandHasBeenSet = false;

  Aws::Vector<BatchEnvironmentVariable> m_environment;
  bool m_environmentHasBeenSet = false;

  Aws::String m_instanceType;
  bool m_instanceTypeHasBeenSet = false;

  Aws::Vector<BatchResourceRequirement> m_resourceRequirements;
  bool m_resourceRequirementsHasBeenSet = false;
};

}  // namespace Model
}  // namespace Pipes
}  // namespace Aws
