﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/redshift/RedshiftRequest.h>
#include <aws/redshift/Redshift_EXPORTS.h>
#include <aws/redshift/model/ReservedNodeExchangeActionType.h>

#include <utility>

namespace Aws {
namespace Redshift {
namespace Model {

/**
 */
class GetReservedNodeExchangeConfigurationOptionsRequest : public RedshiftRequest {
 public:
  AWS_REDSHIFT_API GetReservedNodeExchangeConfigurationOptionsRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "GetReservedNodeExchangeConfigurationOptions"; }

  AWS_REDSHIFT_API Aws::String SerializePayload() const override;

 protected:
  AWS_REDSHIFT_API void DumpBodyToUrl(Aws::Http::URI& uri) const override;

 public:
  ///@{
  /**
   * <p>The action type of the reserved-node configuration. The action type can be an
   * exchange initiated from either a snapshot or a resize.</p>
   */
  inline ReservedNodeExchangeActionType GetActionType() const { return m_actionType; }
  inline bool ActionTypeHasBeenSet() const { return m_actionTypeHasBeenSet; }
  inline void SetActionType(ReservedNodeExchangeActionType value) {
    m_actionTypeHasBeenSet = true;
    m_actionType = value;
  }
  inline GetReservedNodeExchangeConfigurationOptionsRequest& WithActionType(ReservedNodeExchangeActionType value) {
    SetActionType(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The identifier for the cluster that is the source for a reserved-node
   * exchange.</p>
   */
  inline const Aws::String& GetClusterIdentifier() const { return m_clusterIdentifier; }
  inline bool ClusterIdentifierHasBeenSet() const { return m_clusterIdentifierHasBeenSet; }
  template <typename ClusterIdentifierT = Aws::String>
  void SetClusterIdentifier(ClusterIdentifierT&& value) {
    m_clusterIdentifierHasBeenSet = true;
    m_clusterIdentifier = std::forward<ClusterIdentifierT>(value);
  }
  template <typename ClusterIdentifierT = Aws::String>
  GetReservedNodeExchangeConfigurationOptionsRequest& WithClusterIdentifier(ClusterIdentifierT&& value) {
    SetClusterIdentifier(std::forward<ClusterIdentifierT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The identifier for the snapshot that is the source for the reserved-node
   * exchange.</p>
   */
  inline const Aws::String& GetSnapshotIdentifier() const { return m_snapshotIdentifier; }
  inline bool SnapshotIdentifierHasBeenSet() const { return m_snapshotIdentifierHasBeenSet; }
  template <typename SnapshotIdentifierT = Aws::String>
  void SetSnapshotIdentifier(SnapshotIdentifierT&& value) {
    m_snapshotIdentifierHasBeenSet = true;
    m_snapshotIdentifier = std::forward<SnapshotIdentifierT>(value);
  }
  template <typename SnapshotIdentifierT = Aws::String>
  GetReservedNodeExchangeConfigurationOptionsRequest& WithSnapshotIdentifier(SnapshotIdentifierT&& value) {
    SetSnapshotIdentifier(std::forward<SnapshotIdentifierT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of response records to return in each call. If the number
   * of remaining response records exceeds the specified <code>MaxRecords</code>
   * value, a value is returned in a <code>Marker</code> field of the response. You
   * can retrieve the next set of records by retrying the command with the returned
   * marker value.</p>
   */
  inline int GetMaxRecords() const { return m_maxRecords; }
  inline bool MaxRecordsHasBeenSet() const { return m_maxRecordsHasBeenSet; }
  inline void SetMaxRecords(int value) {
    m_maxRecordsHasBeenSet = true;
    m_maxRecords = value;
  }
  inline GetReservedNodeExchangeConfigurationOptionsRequest& WithMaxRecords(int value) {
    SetMaxRecords(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>An optional pagination token provided by a previous
   * <code>GetReservedNodeExchangeConfigurationOptions</code> request. If this
   * parameter is specified, the response includes only records beyond the marker, up
   * to the value specified by the <code>MaxRecords</code> parameter. You can
   * retrieve the next set of response records by providing the returned marker value
   * in the <code>Marker</code> parameter and retrying the request.</p>
   */
  inline const Aws::String& GetMarker() const { return m_marker; }
  inline bool MarkerHasBeenSet() const { return m_markerHasBeenSet; }
  template <typename MarkerT = Aws::String>
  void SetMarker(MarkerT&& value) {
    m_markerHasBeenSet = true;
    m_marker = std::forward<MarkerT>(value);
  }
  template <typename MarkerT = Aws::String>
  GetReservedNodeExchangeConfigurationOptionsRequest& WithMarker(MarkerT&& value) {
    SetMarker(std::forward<MarkerT>(value));
    return *this;
  }
  ///@}
 private:
  ReservedNodeExchangeActionType m_actionType{ReservedNodeExchangeActionType::NOT_SET};
  bool m_actionTypeHasBeenSet = false;

  Aws::String m_clusterIdentifier;
  bool m_clusterIdentifierHasBeenSet = false;

  Aws::String m_snapshotIdentifier;
  bool m_snapshotIdentifierHasBeenSet = false;

  int m_maxRecords{0};
  bool m_maxRecordsHasBeenSet = false;

  Aws::String m_marker;
  bool m_markerHasBeenSet = false;
};

}  // namespace Model
}  // namespace Redshift
}  // namespace Aws
