﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/UUID.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/resiliencehub/ResilienceHubRequest.h>
#include <aws/resiliencehub/ResilienceHub_EXPORTS.h>

#include <utility>

namespace Aws {
namespace ResilienceHub {
namespace Model {

/**
 */
class DeleteAppAssessmentRequest : public ResilienceHubRequest {
 public:
  AWS_RESILIENCEHUB_API DeleteAppAssessmentRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "DeleteAppAssessment"; }

  AWS_RESILIENCEHUB_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>Amazon Resource Name (ARN) of the assessment. The format for this ARN is:
   * arn:<code>partition</code>:resiliencehub:<code>region</code>:<code>account</code>:app-assessment/<code>app-id</code>.
   * For more information about ARNs, see <a
   * href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">
   * Amazon Resource Names (ARNs)</a> in the <i>Amazon Web Services General
   * Reference</i> guide.</p>
   */
  inline const Aws::String& GetAssessmentArn() const { return m_assessmentArn; }
  inline bool AssessmentArnHasBeenSet() const { return m_assessmentArnHasBeenSet; }
  template <typename AssessmentArnT = Aws::String>
  void SetAssessmentArn(AssessmentArnT&& value) {
    m_assessmentArnHasBeenSet = true;
    m_assessmentArn = std::forward<AssessmentArnT>(value);
  }
  template <typename AssessmentArnT = Aws::String>
  DeleteAppAssessmentRequest& WithAssessmentArn(AssessmentArnT&& value) {
    SetAssessmentArn(std::forward<AssessmentArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Used for an idempotency token. A client token is a unique, case-sensitive
   * string of up to 64 ASCII characters. You should not reuse the same client token
   * for other API requests.</p>
   */
  inline const Aws::String& GetClientToken() const { return m_clientToken; }
  inline bool ClientTokenHasBeenSet() const { return m_clientTokenHasBeenSet; }
  template <typename ClientTokenT = Aws::String>
  void SetClientToken(ClientTokenT&& value) {
    m_clientTokenHasBeenSet = true;
    m_clientToken = std::forward<ClientTokenT>(value);
  }
  template <typename ClientTokenT = Aws::String>
  DeleteAppAssessmentRequest& WithClientToken(ClientTokenT&& value) {
    SetClientToken(std::forward<ClientTokenT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_assessmentArn;
  bool m_assessmentArnHasBeenSet = false;

  Aws::String m_clientToken{Aws::Utils::UUID::PseudoRandomUUID()};
  bool m_clientTokenHasBeenSet = true;
};

}  // namespace Model
}  // namespace ResilienceHub
}  // namespace Aws
