﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/sqs/SQSRequest.h>
#include <aws/sqs/SQS_EXPORTS.h>

#include <utility>

namespace Aws {
namespace SQS {
namespace Model {

/**
 * <p/><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/sqs-2012-11-05/RemovePermissionRequest">AWS
 * API Reference</a></p>
 */
class RemovePermissionRequest : public SQSRequest {
 public:
  AWS_SQS_API RemovePermissionRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "RemovePermission"; }

  AWS_SQS_API Aws::String SerializePayload() const override;

  AWS_SQS_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The URL of the Amazon SQS queue from which permissions are removed.</p>
   * <p>Queue URLs and names are case-sensitive.</p>
   */
  inline const Aws::String& GetQueueUrl() const { return m_queueUrl; }
  inline bool QueueUrlHasBeenSet() const { return m_queueUrlHasBeenSet; }
  template <typename QueueUrlT = Aws::String>
  void SetQueueUrl(QueueUrlT&& value) {
    m_queueUrlHasBeenSet = true;
    m_queueUrl = std::forward<QueueUrlT>(value);
  }
  template <typename QueueUrlT = Aws::String>
  RemovePermissionRequest& WithQueueUrl(QueueUrlT&& value) {
    SetQueueUrl(std::forward<QueueUrlT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The identification of the permission to remove. This is the label added using
   * the <code> <a>AddPermission</a> </code> action.</p>
   */
  inline const Aws::String& GetLabel() const { return m_label; }
  inline bool LabelHasBeenSet() const { return m_labelHasBeenSet; }
  template <typename LabelT = Aws::String>
  void SetLabel(LabelT&& value) {
    m_labelHasBeenSet = true;
    m_label = std::forward<LabelT>(value);
  }
  template <typename LabelT = Aws::String>
  RemovePermissionRequest& WithLabel(LabelT&& value) {
    SetLabel(std::forward<LabelT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_queueUrl;
  bool m_queueUrlHasBeenSet = false;

  Aws::String m_label;
  bool m_labelHasBeenSet = false;
};

}  // namespace Model
}  // namespace SQS
}  // namespace Aws
