﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/devops-guru/DevOpsGuruRequest.h>
#include <aws/devops-guru/DevOpsGuru_EXPORTS.h>
#include <aws/devops-guru/model/ResourceCollectionType.h>

#include <utility>

namespace Aws {
namespace Http {
class URI;
}  // namespace Http
namespace DevOpsGuru {
namespace Model {

/**
 */
class GetResourceCollectionRequest : public DevOpsGuruRequest {
 public:
  AWS_DEVOPSGURU_API GetResourceCollectionRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "GetResourceCollection"; }

  AWS_DEVOPSGURU_API Aws::String SerializePayload() const override;

  AWS_DEVOPSGURU_API void AddQueryStringParameters(Aws::Http::URI& uri) const override;

  ///@{
  /**
   * <p> The type of Amazon Web Services resource collections to return. The one
   * valid value is <code>CLOUD_FORMATION</code> for Amazon Web Services
   * CloudFormation stacks. </p>
   */
  inline ResourceCollectionType GetResourceCollectionType() const { return m_resourceCollectionType; }
  inline bool ResourceCollectionTypeHasBeenSet() const { return m_resourceCollectionTypeHasBeenSet; }
  inline void SetResourceCollectionType(ResourceCollectionType value) {
    m_resourceCollectionTypeHasBeenSet = true;
    m_resourceCollectionType = value;
  }
  inline GetResourceCollectionRequest& WithResourceCollectionType(ResourceCollectionType value) {
    SetResourceCollectionType(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The pagination token to use to retrieve the next page of results for this
   * operation. If this value is null, it retrieves the first page.</p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  GetResourceCollectionRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}
 private:
  ResourceCollectionType m_resourceCollectionType{ResourceCollectionType::NOT_SET};
  bool m_resourceCollectionTypeHasBeenSet = false;

  Aws::String m_nextToken;
  bool m_nextTokenHasBeenSet = false;
};

}  // namespace Model
}  // namespace DevOpsGuru
}  // namespace Aws
