﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/iotanalytics/IoTAnalyticsRequest.h>
#include <aws/iotanalytics/IoTAnalytics_EXPORTS.h>
#include <aws/iotanalytics/model/Message.h>

#include <utility>

namespace Aws {
namespace IoTAnalytics {
namespace Model {

/**
 */
class BatchPutMessageRequest : public IoTAnalyticsRequest {
 public:
  AWS_IOTANALYTICS_API BatchPutMessageRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "BatchPutMessage"; }

  AWS_IOTANALYTICS_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The name of the channel where the messages are sent.</p>
   */
  inline const Aws::String& GetChannelName() const { return m_channelName; }
  inline bool ChannelNameHasBeenSet() const { return m_channelNameHasBeenSet; }
  template <typename ChannelNameT = Aws::String>
  void SetChannelName(ChannelNameT&& value) {
    m_channelNameHasBeenSet = true;
    m_channelName = std::forward<ChannelNameT>(value);
  }
  template <typename ChannelNameT = Aws::String>
  BatchPutMessageRequest& WithChannelName(ChannelNameT&& value) {
    SetChannelName(std::forward<ChannelNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The list of messages to be sent. Each message has the format: { "messageId":
   * "string", "payload": "string"}.</p> <p>The field names of message payloads
   * (data) that you send to IoT Analytics:</p> <ul> <li> <p>Must contain only
   * alphanumeric characters and undescores (_). No other special characters are
   * allowed.</p> </li> <li> <p>Must begin with an alphabetic character or single
   * underscore (_).</p> </li> <li> <p>Cannot contain hyphens (-).</p> </li> <li>
   * <p>In regular expression terms:
   * "^[A-Za-z_]([A-Za-z0-9]*|[A-Za-z0-9][A-Za-z0-9_]*)$". </p> </li> <li> <p>Cannot
   * be more than 255 characters.</p> </li> <li> <p>Are case insensitive. (Fields
   * named foo and FOO in the same payload are considered duplicates.)</p> </li>
   * </ul> <p>For example, {"temp_01": 29} or {"_temp_01": 29} are valid, but
   * {"temp-01": 29}, {"01_temp": 29} or {"__temp_01": 29} are invalid in message
   * payloads. </p>
   */
  inline const Aws::Vector<Message>& GetMessages() const { return m_messages; }
  inline bool MessagesHasBeenSet() const { return m_messagesHasBeenSet; }
  template <typename MessagesT = Aws::Vector<Message>>
  void SetMessages(MessagesT&& value) {
    m_messagesHasBeenSet = true;
    m_messages = std::forward<MessagesT>(value);
  }
  template <typename MessagesT = Aws::Vector<Message>>
  BatchPutMessageRequest& WithMessages(MessagesT&& value) {
    SetMessages(std::forward<MessagesT>(value));
    return *this;
  }
  template <typename MessagesT = Message>
  BatchPutMessageRequest& AddMessages(MessagesT&& value) {
    m_messagesHasBeenSet = true;
    m_messages.emplace_back(std::forward<MessagesT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_channelName;
  bool m_channelNameHasBeenSet = false;

  Aws::Vector<Message> m_messages;
  bool m_messagesHasBeenSet = false;
};

}  // namespace Model
}  // namespace IoTAnalytics
}  // namespace Aws
