﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/managedblockchain-query/ManagedBlockchainQueryRequest.h>
#include <aws/managedblockchain-query/ManagedBlockchainQuery_EXPORTS.h>
#include <aws/managedblockchain-query/model/QueryNetwork.h>

#include <utility>

namespace Aws {
namespace ManagedBlockchainQuery {
namespace Model {

/**
 */
class GetTransactionRequest : public ManagedBlockchainQueryRequest {
 public:
  AWS_MANAGEDBLOCKCHAINQUERY_API GetTransactionRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "GetTransaction"; }

  AWS_MANAGEDBLOCKCHAINQUERY_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The hash of a transaction. It is generated when a transaction is created.</p>
   */
  inline const Aws::String& GetTransactionHash() const { return m_transactionHash; }
  inline bool TransactionHashHasBeenSet() const { return m_transactionHashHasBeenSet; }
  template <typename TransactionHashT = Aws::String>
  void SetTransactionHash(TransactionHashT&& value) {
    m_transactionHashHasBeenSet = true;
    m_transactionHash = std::forward<TransactionHashT>(value);
  }
  template <typename TransactionHashT = Aws::String>
  GetTransactionRequest& WithTransactionHash(TransactionHashT&& value) {
    SetTransactionHash(std::forward<TransactionHashT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The identifier of a Bitcoin transaction. It is generated when a transaction
   * is created.</p>  <p> <code>transactionId</code> is only supported on the
   * Bitcoin networks.</p>
   */
  inline const Aws::String& GetTransactionId() const { return m_transactionId; }
  inline bool TransactionIdHasBeenSet() const { return m_transactionIdHasBeenSet; }
  template <typename TransactionIdT = Aws::String>
  void SetTransactionId(TransactionIdT&& value) {
    m_transactionIdHasBeenSet = true;
    m_transactionId = std::forward<TransactionIdT>(value);
  }
  template <typename TransactionIdT = Aws::String>
  GetTransactionRequest& WithTransactionId(TransactionIdT&& value) {
    SetTransactionId(std::forward<TransactionIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The blockchain network where the transaction occurred.</p>
   */
  inline QueryNetwork GetNetwork() const { return m_network; }
  inline bool NetworkHasBeenSet() const { return m_networkHasBeenSet; }
  inline void SetNetwork(QueryNetwork value) {
    m_networkHasBeenSet = true;
    m_network = value;
  }
  inline GetTransactionRequest& WithNetwork(QueryNetwork value) {
    SetNetwork(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_transactionHash;
  bool m_transactionHashHasBeenSet = false;

  Aws::String m_transactionId;
  bool m_transactionIdHasBeenSet = false;

  QueryNetwork m_network{QueryNetwork::NOT_SET};
  bool m_networkHasBeenSet = false;
};

}  // namespace Model
}  // namespace ManagedBlockchainQuery
}  // namespace Aws
