﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/qbusiness/QBusiness_EXPORTS.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace QBusiness {
namespace Model {

/**
 * <p>A set of instructions that define how Amazon Q Business should generate and
 * format responses to user queries. This collection includes parameters for
 * controlling response characteristics such as length, audience targeting,
 * perspective, style, identity, tone, and custom instructions.</p><p><h3>See
 * Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/qbusiness-2023-11-27/InstructionCollection">AWS
 * API Reference</a></p>
 */
class InstructionCollection {
 public:
  AWS_QBUSINESS_API InstructionCollection() = default;
  AWS_QBUSINESS_API InstructionCollection(Aws::Utils::Json::JsonView jsonValue);
  AWS_QBUSINESS_API InstructionCollection& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_QBUSINESS_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>Specifies the desired length of responses generated by Amazon Q Business.
   * This parameter allows administrators to control whether responses are concise
   * and brief or more detailed and comprehensive.</p>
   */
  inline const Aws::String& GetResponseLength() const { return m_responseLength; }
  inline bool ResponseLengthHasBeenSet() const { return m_responseLengthHasBeenSet; }
  template <typename ResponseLengthT = Aws::String>
  void SetResponseLength(ResponseLengthT&& value) {
    m_responseLengthHasBeenSet = true;
    m_responseLength = std::forward<ResponseLengthT>(value);
  }
  template <typename ResponseLengthT = Aws::String>
  InstructionCollection& WithResponseLength(ResponseLengthT&& value) {
    SetResponseLength(std::forward<ResponseLengthT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Defines the intended audience for the responses, allowing Amazon Q Business
   * to tailor its language, terminology, and explanations appropriately. This could
   * range from technical experts to general users with varying levels of domain
   * knowledge.</p>
   */
  inline const Aws::String& GetTargetAudience() const { return m_targetAudience; }
  inline bool TargetAudienceHasBeenSet() const { return m_targetAudienceHasBeenSet; }
  template <typename TargetAudienceT = Aws::String>
  void SetTargetAudience(TargetAudienceT&& value) {
    m_targetAudienceHasBeenSet = true;
    m_targetAudience = std::forward<TargetAudienceT>(value);
  }
  template <typename TargetAudienceT = Aws::String>
  InstructionCollection& WithTargetAudience(TargetAudienceT&& value) {
    SetTargetAudience(std::forward<TargetAudienceT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Determines the point of view or perspective from which Amazon Q Business
   * generates responses, such as first-person, second-person, or third-person
   * perspective, affecting how information is presented to users.</p>
   */
  inline const Aws::String& GetPerspective() const { return m_perspective; }
  inline bool PerspectiveHasBeenSet() const { return m_perspectiveHasBeenSet; }
  template <typename PerspectiveT = Aws::String>
  void SetPerspective(PerspectiveT&& value) {
    m_perspectiveHasBeenSet = true;
    m_perspective = std::forward<PerspectiveT>(value);
  }
  template <typename PerspectiveT = Aws::String>
  InstructionCollection& WithPerspective(PerspectiveT&& value) {
    SetPerspective(std::forward<PerspectiveT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies the formatting and structural style of responses, such as bullet
   * points, paragraphs, step-by-step instructions, or other organizational formats
   * that enhance readability and comprehension.</p>
   */
  inline const Aws::String& GetOutputStyle() const { return m_outputStyle; }
  inline bool OutputStyleHasBeenSet() const { return m_outputStyleHasBeenSet; }
  template <typename OutputStyleT = Aws::String>
  void SetOutputStyle(OutputStyleT&& value) {
    m_outputStyleHasBeenSet = true;
    m_outputStyle = std::forward<OutputStyleT>(value);
  }
  template <typename OutputStyleT = Aws::String>
  InstructionCollection& WithOutputStyle(OutputStyleT&& value) {
    SetOutputStyle(std::forward<OutputStyleT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Defines the persona or identity that Amazon Q Business should adopt when
   * responding to users, allowing for customization of the assistant's character,
   * role, or representation within an organization.</p>
   */
  inline const Aws::String& GetIdentity() const { return m_identity; }
  inline bool IdentityHasBeenSet() const { return m_identityHasBeenSet; }
  template <typename IdentityT = Aws::String>
  void SetIdentity(IdentityT&& value) {
    m_identityHasBeenSet = true;
    m_identity = std::forward<IdentityT>(value);
  }
  template <typename IdentityT = Aws::String>
  InstructionCollection& WithIdentity(IdentityT&& value) {
    SetIdentity(std::forward<IdentityT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Controls the emotional tone and communication style of responses, such as
   * formal, casual, technical, friendly, or professional, to align with
   * organizational communication standards and user expectations.</p>
   */
  inline const Aws::String& GetTone() const { return m_tone; }
  inline bool ToneHasBeenSet() const { return m_toneHasBeenSet; }
  template <typename ToneT = Aws::String>
  void SetTone(ToneT&& value) {
    m_toneHasBeenSet = true;
    m_tone = std::forward<ToneT>(value);
  }
  template <typename ToneT = Aws::String>
  InstructionCollection& WithTone(ToneT&& value) {
    SetTone(std::forward<ToneT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Allows administrators to provide specific, custom instructions that guide how
   * Amazon Q Business should respond in particular scenarios or to certain types of
   * queries, enabling fine-grained control over response generation.</p>
   */
  inline const Aws::String& GetCustomInstructions() const { return m_customInstructions; }
  inline bool CustomInstructionsHasBeenSet() const { return m_customInstructionsHasBeenSet; }
  template <typename CustomInstructionsT = Aws::String>
  void SetCustomInstructions(CustomInstructionsT&& value) {
    m_customInstructionsHasBeenSet = true;
    m_customInstructions = std::forward<CustomInstructionsT>(value);
  }
  template <typename CustomInstructionsT = Aws::String>
  InstructionCollection& WithCustomInstructions(CustomInstructionsT&& value) {
    SetCustomInstructions(std::forward<CustomInstructionsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Provides sample responses or templates that Amazon Q Business can reference
   * when generating responses, helping to establish consistent patterns and formats
   * for different types of user queries.</p>
   */
  inline const Aws::String& GetExamples() const { return m_examples; }
  inline bool ExamplesHasBeenSet() const { return m_examplesHasBeenSet; }
  template <typename ExamplesT = Aws::String>
  void SetExamples(ExamplesT&& value) {
    m_examplesHasBeenSet = true;
    m_examples = std::forward<ExamplesT>(value);
  }
  template <typename ExamplesT = Aws::String>
  InstructionCollection& WithExamples(ExamplesT&& value) {
    SetExamples(std::forward<ExamplesT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_responseLength;
  bool m_responseLengthHasBeenSet = false;

  Aws::String m_targetAudience;
  bool m_targetAudienceHasBeenSet = false;

  Aws::String m_perspective;
  bool m_perspectiveHasBeenSet = false;

  Aws::String m_outputStyle;
  bool m_outputStyleHasBeenSet = false;

  Aws::String m_identity;
  bool m_identityHasBeenSet = false;

  Aws::String m_tone;
  bool m_toneHasBeenSet = false;

  Aws::String m_customInstructions;
  bool m_customInstructionsHasBeenSet = false;

  Aws::String m_examples;
  bool m_examplesHasBeenSet = false;
};

}  // namespace Model
}  // namespace QBusiness
}  // namespace Aws
