﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/qconnect/QConnectRequest.h>
#include <aws/qconnect/QConnect_EXPORTS.h>
#include <aws/qconnect/model/AIAgentConfigurationData.h>
#include <aws/qconnect/model/AIAgentType.h>
#include <aws/qconnect/model/TagFilter.h>

#include <utility>

namespace Aws {
namespace QConnect {
namespace Model {

/**
 */
class UpdateSessionRequest : public QConnectRequest {
 public:
  AWS_QCONNECT_API UpdateSessionRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "UpdateSession"; }

  AWS_QCONNECT_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The identifier of the Amazon Q in Connect assistant. Can be either the ID or
   * the ARN. URLs cannot contain the ARN.</p>
   */
  inline const Aws::String& GetAssistantId() const { return m_assistantId; }
  inline bool AssistantIdHasBeenSet() const { return m_assistantIdHasBeenSet; }
  template <typename AssistantIdT = Aws::String>
  void SetAssistantId(AssistantIdT&& value) {
    m_assistantIdHasBeenSet = true;
    m_assistantId = std::forward<AssistantIdT>(value);
  }
  template <typename AssistantIdT = Aws::String>
  UpdateSessionRequest& WithAssistantId(AssistantIdT&& value) {
    SetAssistantId(std::forward<AssistantIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The identifier of the session. Can be either the ID or the ARN. URLs cannot
   * contain the ARN.</p>
   */
  inline const Aws::String& GetSessionId() const { return m_sessionId; }
  inline bool SessionIdHasBeenSet() const { return m_sessionIdHasBeenSet; }
  template <typename SessionIdT = Aws::String>
  void SetSessionId(SessionIdT&& value) {
    m_sessionIdHasBeenSet = true;
    m_sessionId = std::forward<SessionIdT>(value);
  }
  template <typename SessionIdT = Aws::String>
  UpdateSessionRequest& WithSessionId(SessionIdT&& value) {
    SetSessionId(std::forward<SessionIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The description.</p>
   */
  inline const Aws::String& GetDescription() const { return m_description; }
  inline bool DescriptionHasBeenSet() const { return m_descriptionHasBeenSet; }
  template <typename DescriptionT = Aws::String>
  void SetDescription(DescriptionT&& value) {
    m_descriptionHasBeenSet = true;
    m_description = std::forward<DescriptionT>(value);
  }
  template <typename DescriptionT = Aws::String>
  UpdateSessionRequest& WithDescription(DescriptionT&& value) {
    SetDescription(std::forward<DescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>An object that can be used to specify Tag conditions.</p>
   */
  inline const TagFilter& GetTagFilter() const { return m_tagFilter; }
  inline bool TagFilterHasBeenSet() const { return m_tagFilterHasBeenSet; }
  template <typename TagFilterT = TagFilter>
  void SetTagFilter(TagFilterT&& value) {
    m_tagFilterHasBeenSet = true;
    m_tagFilter = std::forward<TagFilterT>(value);
  }
  template <typename TagFilterT = TagFilter>
  UpdateSessionRequest& WithTagFilter(TagFilterT&& value) {
    SetTagFilter(std::forward<TagFilterT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The configuration of the AI Agents (mapped by AI Agent Type to AI Agent
   * version) that should be used by Amazon Q in Connect for this Session.</p>
   */
  inline const Aws::Map<AIAgentType, AIAgentConfigurationData>& GetAiAgentConfiguration() const { return m_aiAgentConfiguration; }
  inline bool AiAgentConfigurationHasBeenSet() const { return m_aiAgentConfigurationHasBeenSet; }
  template <typename AiAgentConfigurationT = Aws::Map<AIAgentType, AIAgentConfigurationData>>
  void SetAiAgentConfiguration(AiAgentConfigurationT&& value) {
    m_aiAgentConfigurationHasBeenSet = true;
    m_aiAgentConfiguration = std::forward<AiAgentConfigurationT>(value);
  }
  template <typename AiAgentConfigurationT = Aws::Map<AIAgentType, AIAgentConfigurationData>>
  UpdateSessionRequest& WithAiAgentConfiguration(AiAgentConfigurationT&& value) {
    SetAiAgentConfiguration(std::forward<AiAgentConfigurationT>(value));
    return *this;
  }
  inline UpdateSessionRequest& AddAiAgentConfiguration(AIAgentType key, AIAgentConfigurationData value) {
    m_aiAgentConfigurationHasBeenSet = true;
    m_aiAgentConfiguration.emplace(key, value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_assistantId;
  bool m_assistantIdHasBeenSet = false;

  Aws::String m_sessionId;
  bool m_sessionIdHasBeenSet = false;

  Aws::String m_description;
  bool m_descriptionHasBeenSet = false;

  TagFilter m_tagFilter;
  bool m_tagFilterHasBeenSet = false;

  Aws::Map<AIAgentType, AIAgentConfigurationData> m_aiAgentConfiguration;
  bool m_aiAgentConfigurationHasBeenSet = false;
};

}  // namespace Model
}  // namespace QConnect
}  // namespace Aws
