﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/DateTime.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/ssm/SSM_EXPORTS.h>
#include <aws/ssm/model/DocumentFormat.h>
#include <aws/ssm/model/DocumentRequires.h>
#include <aws/ssm/model/DocumentType.h>
#include <aws/ssm/model/PlatformType.h>
#include <aws/ssm/model/ReviewStatus.h>
#include <aws/ssm/model/Tag.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace SSM {
namespace Model {

/**
 * <p>Describes the name of a SSM document.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/ssm-2014-11-06/DocumentIdentifier">AWS
 * API Reference</a></p>
 */
class DocumentIdentifier {
 public:
  AWS_SSM_API DocumentIdentifier() = default;
  AWS_SSM_API DocumentIdentifier(Aws::Utils::Json::JsonView jsonValue);
  AWS_SSM_API DocumentIdentifier& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_SSM_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>The name of the SSM document.</p>
   */
  inline const Aws::String& GetName() const { return m_name; }
  inline bool NameHasBeenSet() const { return m_nameHasBeenSet; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  DocumentIdentifier& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The date the SSM document was created.</p>
   */
  inline const Aws::Utils::DateTime& GetCreatedDate() const { return m_createdDate; }
  inline bool CreatedDateHasBeenSet() const { return m_createdDateHasBeenSet; }
  template <typename CreatedDateT = Aws::Utils::DateTime>
  void SetCreatedDate(CreatedDateT&& value) {
    m_createdDateHasBeenSet = true;
    m_createdDate = std::forward<CreatedDateT>(value);
  }
  template <typename CreatedDateT = Aws::Utils::DateTime>
  DocumentIdentifier& WithCreatedDate(CreatedDateT&& value) {
    SetCreatedDate(std::forward<CreatedDateT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>An optional field where you can specify a friendly name for the SSM document.
   * This value can differ for each version of the document. If you want to update
   * this value, see <a>UpdateDocument</a>.</p>
   */
  inline const Aws::String& GetDisplayName() const { return m_displayName; }
  inline bool DisplayNameHasBeenSet() const { return m_displayNameHasBeenSet; }
  template <typename DisplayNameT = Aws::String>
  void SetDisplayName(DisplayNameT&& value) {
    m_displayNameHasBeenSet = true;
    m_displayName = std::forward<DisplayNameT>(value);
  }
  template <typename DisplayNameT = Aws::String>
  DocumentIdentifier& WithDisplayName(DisplayNameT&& value) {
    SetDisplayName(std::forward<DisplayNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Web Services user that created the document.</p>
   */
  inline const Aws::String& GetOwner() const { return m_owner; }
  inline bool OwnerHasBeenSet() const { return m_ownerHasBeenSet; }
  template <typename OwnerT = Aws::String>
  void SetOwner(OwnerT&& value) {
    m_ownerHasBeenSet = true;
    m_owner = std::forward<OwnerT>(value);
  }
  template <typename OwnerT = Aws::String>
  DocumentIdentifier& WithOwner(OwnerT&& value) {
    SetOwner(std::forward<OwnerT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>An optional field specifying the version of the artifact associated with the
   * document. For example, 12.6. This value is unique across all versions of a
   * document, and can't be changed.</p>
   */
  inline const Aws::String& GetVersionName() const { return m_versionName; }
  inline bool VersionNameHasBeenSet() const { return m_versionNameHasBeenSet; }
  template <typename VersionNameT = Aws::String>
  void SetVersionName(VersionNameT&& value) {
    m_versionNameHasBeenSet = true;
    m_versionName = std::forward<VersionNameT>(value);
  }
  template <typename VersionNameT = Aws::String>
  DocumentIdentifier& WithVersionName(VersionNameT&& value) {
    SetVersionName(std::forward<VersionNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The operating system platform. </p>
   */
  inline const Aws::Vector<PlatformType>& GetPlatformTypes() const { return m_platformTypes; }
  inline bool PlatformTypesHasBeenSet() const { return m_platformTypesHasBeenSet; }
  template <typename PlatformTypesT = Aws::Vector<PlatformType>>
  void SetPlatformTypes(PlatformTypesT&& value) {
    m_platformTypesHasBeenSet = true;
    m_platformTypes = std::forward<PlatformTypesT>(value);
  }
  template <typename PlatformTypesT = Aws::Vector<PlatformType>>
  DocumentIdentifier& WithPlatformTypes(PlatformTypesT&& value) {
    SetPlatformTypes(std::forward<PlatformTypesT>(value));
    return *this;
  }
  inline DocumentIdentifier& AddPlatformTypes(PlatformType value) {
    m_platformTypesHasBeenSet = true;
    m_platformTypes.push_back(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The document version.</p>
   */
  inline const Aws::String& GetDocumentVersion() const { return m_documentVersion; }
  inline bool DocumentVersionHasBeenSet() const { return m_documentVersionHasBeenSet; }
  template <typename DocumentVersionT = Aws::String>
  void SetDocumentVersion(DocumentVersionT&& value) {
    m_documentVersionHasBeenSet = true;
    m_documentVersion = std::forward<DocumentVersionT>(value);
  }
  template <typename DocumentVersionT = Aws::String>
  DocumentIdentifier& WithDocumentVersion(DocumentVersionT&& value) {
    SetDocumentVersion(std::forward<DocumentVersionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The document type.</p>
   */
  inline DocumentType GetDocumentType() const { return m_documentType; }
  inline bool DocumentTypeHasBeenSet() const { return m_documentTypeHasBeenSet; }
  inline void SetDocumentType(DocumentType value) {
    m_documentTypeHasBeenSet = true;
    m_documentType = value;
  }
  inline DocumentIdentifier& WithDocumentType(DocumentType value) {
    SetDocumentType(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The schema version.</p>
   */
  inline const Aws::String& GetSchemaVersion() const { return m_schemaVersion; }
  inline bool SchemaVersionHasBeenSet() const { return m_schemaVersionHasBeenSet; }
  template <typename SchemaVersionT = Aws::String>
  void SetSchemaVersion(SchemaVersionT&& value) {
    m_schemaVersionHasBeenSet = true;
    m_schemaVersion = std::forward<SchemaVersionT>(value);
  }
  template <typename SchemaVersionT = Aws::String>
  DocumentIdentifier& WithSchemaVersion(SchemaVersionT&& value) {
    SetSchemaVersion(std::forward<SchemaVersionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The document format, either JSON or YAML.</p>
   */
  inline DocumentFormat GetDocumentFormat() const { return m_documentFormat; }
  inline bool DocumentFormatHasBeenSet() const { return m_documentFormatHasBeenSet; }
  inline void SetDocumentFormat(DocumentFormat value) {
    m_documentFormatHasBeenSet = true;
    m_documentFormat = value;
  }
  inline DocumentIdentifier& WithDocumentFormat(DocumentFormat value) {
    SetDocumentFormat(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The target type which defines the kinds of resources the document can run on.
   * For example, <code>/AWS::EC2::Instance</code>. For a list of valid resource
   * types, see <a
   * href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-template-resource-type-ref.html">Amazon
   * Web Services resource and property types reference</a> in the <i>CloudFormation
   * User Guide</i>. </p>
   */
  inline const Aws::String& GetTargetType() const { return m_targetType; }
  inline bool TargetTypeHasBeenSet() const { return m_targetTypeHasBeenSet; }
  template <typename TargetTypeT = Aws::String>
  void SetTargetType(TargetTypeT&& value) {
    m_targetTypeHasBeenSet = true;
    m_targetType = std::forward<TargetTypeT>(value);
  }
  template <typename TargetTypeT = Aws::String>
  DocumentIdentifier& WithTargetType(TargetTypeT&& value) {
    SetTargetType(std::forward<TargetTypeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The tags, or metadata, that have been applied to the document.</p>
   */
  inline const Aws::Vector<Tag>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Vector<Tag>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Vector<Tag>>
  DocumentIdentifier& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsT = Tag>
  DocumentIdentifier& AddTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace_back(std::forward<TagsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A list of SSM documents required by a document. For example, an
   * <code>ApplicationConfiguration</code> document requires an
   * <code>ApplicationConfigurationSchema</code> document.</p>
   */
  inline const Aws::Vector<DocumentRequires>& GetRequires() const { return m_requires; }
  inline bool RequiresHasBeenSet() const { return m_requiresHasBeenSet; }
  template <typename RequiresT = Aws::Vector<DocumentRequires>>
  void SetRequires(RequiresT&& value) {
    m_requiresHasBeenSet = true;
    m_requires = std::forward<RequiresT>(value);
  }
  template <typename RequiresT = Aws::Vector<DocumentRequires>>
  DocumentIdentifier& WithRequires(RequiresT&& value) {
    SetRequires(std::forward<RequiresT>(value));
    return *this;
  }
  template <typename RequiresT = DocumentRequires>
  DocumentIdentifier& AddRequires(RequiresT&& value) {
    m_requiresHasBeenSet = true;
    m_requires.emplace_back(std::forward<RequiresT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The current status of a document review.</p>
   */
  inline ReviewStatus GetReviewStatus() const { return m_reviewStatus; }
  inline bool ReviewStatusHasBeenSet() const { return m_reviewStatusHasBeenSet; }
  inline void SetReviewStatus(ReviewStatus value) {
    m_reviewStatusHasBeenSet = true;
    m_reviewStatus = value;
  }
  inline DocumentIdentifier& WithReviewStatus(ReviewStatus value) {
    SetReviewStatus(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The user in your organization who created the document.</p>
   */
  inline const Aws::String& GetAuthor() const { return m_author; }
  inline bool AuthorHasBeenSet() const { return m_authorHasBeenSet; }
  template <typename AuthorT = Aws::String>
  void SetAuthor(AuthorT&& value) {
    m_authorHasBeenSet = true;
    m_author = std::forward<AuthorT>(value);
  }
  template <typename AuthorT = Aws::String>
  DocumentIdentifier& WithAuthor(AuthorT&& value) {
    SetAuthor(std::forward<AuthorT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_name;
  bool m_nameHasBeenSet = false;

  Aws::Utils::DateTime m_createdDate{};
  bool m_createdDateHasBeenSet = false;

  Aws::String m_displayName;
  bool m_displayNameHasBeenSet = false;

  Aws::String m_owner;
  bool m_ownerHasBeenSet = false;

  Aws::String m_versionName;
  bool m_versionNameHasBeenSet = false;

  Aws::Vector<PlatformType> m_platformTypes;
  bool m_platformTypesHasBeenSet = false;

  Aws::String m_documentVersion;
  bool m_documentVersionHasBeenSet = false;

  DocumentType m_documentType{DocumentType::NOT_SET};
  bool m_documentTypeHasBeenSet = false;

  Aws::String m_schemaVersion;
  bool m_schemaVersionHasBeenSet = false;

  DocumentFormat m_documentFormat{DocumentFormat::NOT_SET};
  bool m_documentFormatHasBeenSet = false;

  Aws::String m_targetType;
  bool m_targetTypeHasBeenSet = false;

  Aws::Vector<Tag> m_tags;
  bool m_tagsHasBeenSet = false;

  Aws::Vector<DocumentRequires> m_requires;
  bool m_requiresHasBeenSet = false;

  ReviewStatus m_reviewStatus{ReviewStatus::NOT_SET};
  bool m_reviewStatusHasBeenSet = false;

  Aws::String m_author;
  bool m_authorHasBeenSet = false;
};

}  // namespace Model
}  // namespace SSM
}  // namespace Aws
