﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/dax/DAXRequest.h>
#include <aws/dax/DAX_EXPORTS.h>

#include <utility>

namespace Aws {
namespace DAX {
namespace Model {

/**
 */
class UpdateClusterRequest : public DAXRequest {
 public:
  AWS_DAX_API UpdateClusterRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "UpdateCluster"; }

  AWS_DAX_API Aws::String SerializePayload() const override;

  AWS_DAX_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The name of the DAX cluster to be modified.</p>
   */
  inline const Aws::String& GetClusterName() const { return m_clusterName; }
  inline bool ClusterNameHasBeenSet() const { return m_clusterNameHasBeenSet; }
  template <typename ClusterNameT = Aws::String>
  void SetClusterName(ClusterNameT&& value) {
    m_clusterNameHasBeenSet = true;
    m_clusterName = std::forward<ClusterNameT>(value);
  }
  template <typename ClusterNameT = Aws::String>
  UpdateClusterRequest& WithClusterName(ClusterNameT&& value) {
    SetClusterName(std::forward<ClusterNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A description of the changes being made to the cluster.</p>
   */
  inline const Aws::String& GetDescription() const { return m_description; }
  inline bool DescriptionHasBeenSet() const { return m_descriptionHasBeenSet; }
  template <typename DescriptionT = Aws::String>
  void SetDescription(DescriptionT&& value) {
    m_descriptionHasBeenSet = true;
    m_description = std::forward<DescriptionT>(value);
  }
  template <typename DescriptionT = Aws::String>
  UpdateClusterRequest& WithDescription(DescriptionT&& value) {
    SetDescription(std::forward<DescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A range of time when maintenance of DAX cluster software will be performed.
   * For example: <code>sun:01:00-sun:09:00</code>. Cluster maintenance normally
   * takes less than 30 minutes, and is performed automatically within the
   * maintenance window.</p>
   */
  inline const Aws::String& GetPreferredMaintenanceWindow() const { return m_preferredMaintenanceWindow; }
  inline bool PreferredMaintenanceWindowHasBeenSet() const { return m_preferredMaintenanceWindowHasBeenSet; }
  template <typename PreferredMaintenanceWindowT = Aws::String>
  void SetPreferredMaintenanceWindow(PreferredMaintenanceWindowT&& value) {
    m_preferredMaintenanceWindowHasBeenSet = true;
    m_preferredMaintenanceWindow = std::forward<PreferredMaintenanceWindowT>(value);
  }
  template <typename PreferredMaintenanceWindowT = Aws::String>
  UpdateClusterRequest& WithPreferredMaintenanceWindow(PreferredMaintenanceWindowT&& value) {
    SetPreferredMaintenanceWindow(std::forward<PreferredMaintenanceWindowT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) that identifies the topic.</p>
   */
  inline const Aws::String& GetNotificationTopicArn() const { return m_notificationTopicArn; }
  inline bool NotificationTopicArnHasBeenSet() const { return m_notificationTopicArnHasBeenSet; }
  template <typename NotificationTopicArnT = Aws::String>
  void SetNotificationTopicArn(NotificationTopicArnT&& value) {
    m_notificationTopicArnHasBeenSet = true;
    m_notificationTopicArn = std::forward<NotificationTopicArnT>(value);
  }
  template <typename NotificationTopicArnT = Aws::String>
  UpdateClusterRequest& WithNotificationTopicArn(NotificationTopicArnT&& value) {
    SetNotificationTopicArn(std::forward<NotificationTopicArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The current state of the topic. A value of “active” means that notifications
   * will be sent to the topic. A value of “inactive” means that notifications will
   * not be sent to the topic.</p>
   */
  inline const Aws::String& GetNotificationTopicStatus() const { return m_notificationTopicStatus; }
  inline bool NotificationTopicStatusHasBeenSet() const { return m_notificationTopicStatusHasBeenSet; }
  template <typename NotificationTopicStatusT = Aws::String>
  void SetNotificationTopicStatus(NotificationTopicStatusT&& value) {
    m_notificationTopicStatusHasBeenSet = true;
    m_notificationTopicStatus = std::forward<NotificationTopicStatusT>(value);
  }
  template <typename NotificationTopicStatusT = Aws::String>
  UpdateClusterRequest& WithNotificationTopicStatus(NotificationTopicStatusT&& value) {
    SetNotificationTopicStatus(std::forward<NotificationTopicStatusT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of a parameter group for this cluster.</p>
   */
  inline const Aws::String& GetParameterGroupName() const { return m_parameterGroupName; }
  inline bool ParameterGroupNameHasBeenSet() const { return m_parameterGroupNameHasBeenSet; }
  template <typename ParameterGroupNameT = Aws::String>
  void SetParameterGroupName(ParameterGroupNameT&& value) {
    m_parameterGroupNameHasBeenSet = true;
    m_parameterGroupName = std::forward<ParameterGroupNameT>(value);
  }
  template <typename ParameterGroupNameT = Aws::String>
  UpdateClusterRequest& WithParameterGroupName(ParameterGroupNameT&& value) {
    SetParameterGroupName(std::forward<ParameterGroupNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A list of user-specified security group IDs to be assigned to each node in
   * the DAX cluster. If this parameter is not specified, DAX assigns the default VPC
   * security group to each node.</p>
   */
  inline const Aws::Vector<Aws::String>& GetSecurityGroupIds() const { return m_securityGroupIds; }
  inline bool SecurityGroupIdsHasBeenSet() const { return m_securityGroupIdsHasBeenSet; }
  template <typename SecurityGroupIdsT = Aws::Vector<Aws::String>>
  void SetSecurityGroupIds(SecurityGroupIdsT&& value) {
    m_securityGroupIdsHasBeenSet = true;
    m_securityGroupIds = std::forward<SecurityGroupIdsT>(value);
  }
  template <typename SecurityGroupIdsT = Aws::Vector<Aws::String>>
  UpdateClusterRequest& WithSecurityGroupIds(SecurityGroupIdsT&& value) {
    SetSecurityGroupIds(std::forward<SecurityGroupIdsT>(value));
    return *this;
  }
  template <typename SecurityGroupIdsT = Aws::String>
  UpdateClusterRequest& AddSecurityGroupIds(SecurityGroupIdsT&& value) {
    m_securityGroupIdsHasBeenSet = true;
    m_securityGroupIds.emplace_back(std::forward<SecurityGroupIdsT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_clusterName;
  bool m_clusterNameHasBeenSet = false;

  Aws::String m_description;
  bool m_descriptionHasBeenSet = false;

  Aws::String m_preferredMaintenanceWindow;
  bool m_preferredMaintenanceWindowHasBeenSet = false;

  Aws::String m_notificationTopicArn;
  bool m_notificationTopicArnHasBeenSet = false;

  Aws::String m_notificationTopicStatus;
  bool m_notificationTopicStatusHasBeenSet = false;

  Aws::String m_parameterGroupName;
  bool m_parameterGroupNameHasBeenSet = false;

  Aws::Vector<Aws::String> m_securityGroupIds;
  bool m_securityGroupIdsHasBeenSet = false;
};

}  // namespace Model
}  // namespace DAX
}  // namespace Aws
