﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/DateTime.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/emr-containers/EMRContainersRequest.h>
#include <aws/emr-containers/EMRContainers_EXPORTS.h>
#include <aws/emr-containers/model/EndpointState.h>

#include <utility>

namespace Aws {
namespace Http {
class URI;
}  // namespace Http
namespace EMRContainers {
namespace Model {

/**
 */
class ListManagedEndpointsRequest : public EMRContainersRequest {
 public:
  AWS_EMRCONTAINERS_API ListManagedEndpointsRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ListManagedEndpoints"; }

  AWS_EMRCONTAINERS_API Aws::String SerializePayload() const override;

  AWS_EMRCONTAINERS_API void AddQueryStringParameters(Aws::Http::URI& uri) const override;

  ///@{
  /**
   * <p>The ID of the virtual cluster.</p>
   */
  inline const Aws::String& GetVirtualClusterId() const { return m_virtualClusterId; }
  inline bool VirtualClusterIdHasBeenSet() const { return m_virtualClusterIdHasBeenSet; }
  template <typename VirtualClusterIdT = Aws::String>
  void SetVirtualClusterId(VirtualClusterIdT&& value) {
    m_virtualClusterIdHasBeenSet = true;
    m_virtualClusterId = std::forward<VirtualClusterIdT>(value);
  }
  template <typename VirtualClusterIdT = Aws::String>
  ListManagedEndpointsRequest& WithVirtualClusterId(VirtualClusterIdT&& value) {
    SetVirtualClusterId(std::forward<VirtualClusterIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The date and time before which the endpoints are created.</p>
   */
  inline const Aws::Utils::DateTime& GetCreatedBefore() const { return m_createdBefore; }
  inline bool CreatedBeforeHasBeenSet() const { return m_createdBeforeHasBeenSet; }
  template <typename CreatedBeforeT = Aws::Utils::DateTime>
  void SetCreatedBefore(CreatedBeforeT&& value) {
    m_createdBeforeHasBeenSet = true;
    m_createdBefore = std::forward<CreatedBeforeT>(value);
  }
  template <typename CreatedBeforeT = Aws::Utils::DateTime>
  ListManagedEndpointsRequest& WithCreatedBefore(CreatedBeforeT&& value) {
    SetCreatedBefore(std::forward<CreatedBeforeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> The date and time after which the endpoints are created.</p>
   */
  inline const Aws::Utils::DateTime& GetCreatedAfter() const { return m_createdAfter; }
  inline bool CreatedAfterHasBeenSet() const { return m_createdAfterHasBeenSet; }
  template <typename CreatedAfterT = Aws::Utils::DateTime>
  void SetCreatedAfter(CreatedAfterT&& value) {
    m_createdAfterHasBeenSet = true;
    m_createdAfter = std::forward<CreatedAfterT>(value);
  }
  template <typename CreatedAfterT = Aws::Utils::DateTime>
  ListManagedEndpointsRequest& WithCreatedAfter(CreatedAfterT&& value) {
    SetCreatedAfter(std::forward<CreatedAfterT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The types of the managed endpoints.</p>
   */
  inline const Aws::Vector<Aws::String>& GetTypes() const { return m_types; }
  inline bool TypesHasBeenSet() const { return m_typesHasBeenSet; }
  template <typename TypesT = Aws::Vector<Aws::String>>
  void SetTypes(TypesT&& value) {
    m_typesHasBeenSet = true;
    m_types = std::forward<TypesT>(value);
  }
  template <typename TypesT = Aws::Vector<Aws::String>>
  ListManagedEndpointsRequest& WithTypes(TypesT&& value) {
    SetTypes(std::forward<TypesT>(value));
    return *this;
  }
  template <typename TypesT = Aws::String>
  ListManagedEndpointsRequest& AddTypes(TypesT&& value) {
    m_typesHasBeenSet = true;
    m_types.emplace_back(std::forward<TypesT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The states of the managed endpoints.</p>
   */
  inline const Aws::Vector<EndpointState>& GetStates() const { return m_states; }
  inline bool StatesHasBeenSet() const { return m_statesHasBeenSet; }
  template <typename StatesT = Aws::Vector<EndpointState>>
  void SetStates(StatesT&& value) {
    m_statesHasBeenSet = true;
    m_states = std::forward<StatesT>(value);
  }
  template <typename StatesT = Aws::Vector<EndpointState>>
  ListManagedEndpointsRequest& WithStates(StatesT&& value) {
    SetStates(std::forward<StatesT>(value));
    return *this;
  }
  inline ListManagedEndpointsRequest& AddStates(EndpointState value) {
    m_statesHasBeenSet = true;
    m_states.push_back(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of managed endpoints that can be listed.</p>
   */
  inline int GetMaxResults() const { return m_maxResults; }
  inline bool MaxResultsHasBeenSet() const { return m_maxResultsHasBeenSet; }
  inline void SetMaxResults(int value) {
    m_maxResultsHasBeenSet = true;
    m_maxResults = value;
  }
  inline ListManagedEndpointsRequest& WithMaxResults(int value) {
    SetMaxResults(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> The token for the next set of managed endpoints to return. </p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  ListManagedEndpointsRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_virtualClusterId;
  bool m_virtualClusterIdHasBeenSet = false;

  Aws::Utils::DateTime m_createdBefore{};
  bool m_createdBeforeHasBeenSet = false;

  Aws::Utils::DateTime m_createdAfter{};
  bool m_createdAfterHasBeenSet = false;

  Aws::Vector<Aws::String> m_types;
  bool m_typesHasBeenSet = false;

  Aws::Vector<EndpointState> m_states;
  bool m_statesHasBeenSet = false;

  int m_maxResults{0};
  bool m_maxResultsHasBeenSet = false;

  Aws::String m_nextToken;
  bool m_nextTokenHasBeenSet = false;
};

}  // namespace Model
}  // namespace EMRContainers
}  // namespace Aws
