﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/UUID.h>
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/finspace/FinspaceRequest.h>
#include <aws/finspace/Finspace_EXPORTS.h>
#include <aws/finspace/model/KxAzMode.h>
#include <aws/finspace/model/KxNAS1Configuration.h>
#include <aws/finspace/model/KxVolumeType.h>

#include <utility>

namespace Aws {
namespace finspace {
namespace Model {

/**
 */
class CreateKxVolumeRequest : public FinspaceRequest {
 public:
  AWS_FINSPACE_API CreateKxVolumeRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateKxVolume"; }

  AWS_FINSPACE_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>A token that ensures idempotency. This token expires in 10 minutes.</p>
   */
  inline const Aws::String& GetClientToken() const { return m_clientToken; }
  inline bool ClientTokenHasBeenSet() const { return m_clientTokenHasBeenSet; }
  template <typename ClientTokenT = Aws::String>
  void SetClientToken(ClientTokenT&& value) {
    m_clientTokenHasBeenSet = true;
    m_clientToken = std::forward<ClientTokenT>(value);
  }
  template <typename ClientTokenT = Aws::String>
  CreateKxVolumeRequest& WithClientToken(ClientTokenT&& value) {
    SetClientToken(std::forward<ClientTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A unique identifier for the kdb environment, whose clusters can attach to the
   * volume. </p>
   */
  inline const Aws::String& GetEnvironmentId() const { return m_environmentId; }
  inline bool EnvironmentIdHasBeenSet() const { return m_environmentIdHasBeenSet; }
  template <typename EnvironmentIdT = Aws::String>
  void SetEnvironmentId(EnvironmentIdT&& value) {
    m_environmentIdHasBeenSet = true;
    m_environmentId = std::forward<EnvironmentIdT>(value);
  }
  template <typename EnvironmentIdT = Aws::String>
  CreateKxVolumeRequest& WithEnvironmentId(EnvironmentIdT&& value) {
    SetEnvironmentId(std::forward<EnvironmentIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> The type of file system volume. Currently, FinSpace only supports
   * <code>NAS_1</code> volume type. When you select <code>NAS_1</code> volume type,
   * you must also provide <code>nas1Configuration</code>. </p>
   */
  inline KxVolumeType GetVolumeType() const { return m_volumeType; }
  inline bool VolumeTypeHasBeenSet() const { return m_volumeTypeHasBeenSet; }
  inline void SetVolumeType(KxVolumeType value) {
    m_volumeTypeHasBeenSet = true;
    m_volumeType = value;
  }
  inline CreateKxVolumeRequest& WithVolumeType(KxVolumeType value) {
    SetVolumeType(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A unique identifier for the volume.</p>
   */
  inline const Aws::String& GetVolumeName() const { return m_volumeName; }
  inline bool VolumeNameHasBeenSet() const { return m_volumeNameHasBeenSet; }
  template <typename VolumeNameT = Aws::String>
  void SetVolumeName(VolumeNameT&& value) {
    m_volumeNameHasBeenSet = true;
    m_volumeName = std::forward<VolumeNameT>(value);
  }
  template <typename VolumeNameT = Aws::String>
  CreateKxVolumeRequest& WithVolumeName(VolumeNameT&& value) {
    SetVolumeName(std::forward<VolumeNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> A description of the volume. </p>
   */
  inline const Aws::String& GetDescription() const { return m_description; }
  inline bool DescriptionHasBeenSet() const { return m_descriptionHasBeenSet; }
  template <typename DescriptionT = Aws::String>
  void SetDescription(DescriptionT&& value) {
    m_descriptionHasBeenSet = true;
    m_description = std::forward<DescriptionT>(value);
  }
  template <typename DescriptionT = Aws::String>
  CreateKxVolumeRequest& WithDescription(DescriptionT&& value) {
    SetDescription(std::forward<DescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> Specifies the configuration for the Network attached storage (NAS_1) file
   * system volume. This parameter is required when you choose
   * <code>volumeType</code> as <i>NAS_1</i>.</p>
   */
  inline const KxNAS1Configuration& GetNas1Configuration() const { return m_nas1Configuration; }
  inline bool Nas1ConfigurationHasBeenSet() const { return m_nas1ConfigurationHasBeenSet; }
  template <typename Nas1ConfigurationT = KxNAS1Configuration>
  void SetNas1Configuration(Nas1ConfigurationT&& value) {
    m_nas1ConfigurationHasBeenSet = true;
    m_nas1Configuration = std::forward<Nas1ConfigurationT>(value);
  }
  template <typename Nas1ConfigurationT = KxNAS1Configuration>
  CreateKxVolumeRequest& WithNas1Configuration(Nas1ConfigurationT&& value) {
    SetNas1Configuration(std::forward<Nas1ConfigurationT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The number of availability zones you want to assign per volume. Currently,
   * FinSpace only supports <code>SINGLE</code> for volumes. This places dataview in
   * a single AZ.</p>
   */
  inline KxAzMode GetAzMode() const { return m_azMode; }
  inline bool AzModeHasBeenSet() const { return m_azModeHasBeenSet; }
  inline void SetAzMode(KxAzMode value) {
    m_azModeHasBeenSet = true;
    m_azMode = value;
  }
  inline CreateKxVolumeRequest& WithAzMode(KxAzMode value) {
    SetAzMode(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The identifier of the availability zones.</p>
   */
  inline const Aws::Vector<Aws::String>& GetAvailabilityZoneIds() const { return m_availabilityZoneIds; }
  inline bool AvailabilityZoneIdsHasBeenSet() const { return m_availabilityZoneIdsHasBeenSet; }
  template <typename AvailabilityZoneIdsT = Aws::Vector<Aws::String>>
  void SetAvailabilityZoneIds(AvailabilityZoneIdsT&& value) {
    m_availabilityZoneIdsHasBeenSet = true;
    m_availabilityZoneIds = std::forward<AvailabilityZoneIdsT>(value);
  }
  template <typename AvailabilityZoneIdsT = Aws::Vector<Aws::String>>
  CreateKxVolumeRequest& WithAvailabilityZoneIds(AvailabilityZoneIdsT&& value) {
    SetAvailabilityZoneIds(std::forward<AvailabilityZoneIdsT>(value));
    return *this;
  }
  template <typename AvailabilityZoneIdsT = Aws::String>
  CreateKxVolumeRequest& AddAvailabilityZoneIds(AvailabilityZoneIdsT&& value) {
    m_availabilityZoneIdsHasBeenSet = true;
    m_availabilityZoneIds.emplace_back(std::forward<AvailabilityZoneIdsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> A list of key-value pairs to label the volume. You can add up to 50 tags to
   * a volume. </p>
   */
  inline const Aws::Map<Aws::String, Aws::String>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  CreateKxVolumeRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsKeyT = Aws::String, typename TagsValueT = Aws::String>
  CreateKxVolumeRequest& AddTags(TagsKeyT&& key, TagsValueT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace(std::forward<TagsKeyT>(key), std::forward<TagsValueT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_clientToken{Aws::Utils::UUID::PseudoRandomUUID()};
  bool m_clientTokenHasBeenSet = true;

  Aws::String m_environmentId;
  bool m_environmentIdHasBeenSet = false;

  KxVolumeType m_volumeType{KxVolumeType::NOT_SET};
  bool m_volumeTypeHasBeenSet = false;

  Aws::String m_volumeName;
  bool m_volumeNameHasBeenSet = false;

  Aws::String m_description;
  bool m_descriptionHasBeenSet = false;

  KxNAS1Configuration m_nas1Configuration;
  bool m_nas1ConfigurationHasBeenSet = false;

  KxAzMode m_azMode{KxAzMode::NOT_SET};
  bool m_azModeHasBeenSet = false;

  Aws::Vector<Aws::String> m_availabilityZoneIds;
  bool m_availabilityZoneIdsHasBeenSet = false;

  Aws::Map<Aws::String, Aws::String> m_tags;
  bool m_tagsHasBeenSet = false;
};

}  // namespace Model
}  // namespace finspace
}  // namespace Aws
