﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/Array.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/lambda/Lambda_EXPORTS.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace Lambda {
namespace Model {

/**
 * <p>A ZIP archive that contains the contents of an <a
 * href="https://docs.aws.amazon.com/lambda/latest/dg/configuration-layers.html">Lambda
 * layer</a>. You can specify either an Amazon S3 location, or upload a layer
 * archive directly.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/lambda-2015-03-31/LayerVersionContentInput">AWS
 * API Reference</a></p>
 */
class LayerVersionContentInput {
 public:
  AWS_LAMBDA_API LayerVersionContentInput() = default;
  AWS_LAMBDA_API LayerVersionContentInput(Aws::Utils::Json::JsonView jsonValue);
  AWS_LAMBDA_API LayerVersionContentInput& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_LAMBDA_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>The Amazon S3 bucket of the layer archive.</p>
   */
  inline const Aws::String& GetS3Bucket() const { return m_s3Bucket; }
  inline bool S3BucketHasBeenSet() const { return m_s3BucketHasBeenSet; }
  template <typename S3BucketT = Aws::String>
  void SetS3Bucket(S3BucketT&& value) {
    m_s3BucketHasBeenSet = true;
    m_s3Bucket = std::forward<S3BucketT>(value);
  }
  template <typename S3BucketT = Aws::String>
  LayerVersionContentInput& WithS3Bucket(S3BucketT&& value) {
    SetS3Bucket(std::forward<S3BucketT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon S3 key of the layer archive.</p>
   */
  inline const Aws::String& GetS3Key() const { return m_s3Key; }
  inline bool S3KeyHasBeenSet() const { return m_s3KeyHasBeenSet; }
  template <typename S3KeyT = Aws::String>
  void SetS3Key(S3KeyT&& value) {
    m_s3KeyHasBeenSet = true;
    m_s3Key = std::forward<S3KeyT>(value);
  }
  template <typename S3KeyT = Aws::String>
  LayerVersionContentInput& WithS3Key(S3KeyT&& value) {
    SetS3Key(std::forward<S3KeyT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>For versioned objects, the version of the layer archive object to use.</p>
   */
  inline const Aws::String& GetS3ObjectVersion() const { return m_s3ObjectVersion; }
  inline bool S3ObjectVersionHasBeenSet() const { return m_s3ObjectVersionHasBeenSet; }
  template <typename S3ObjectVersionT = Aws::String>
  void SetS3ObjectVersion(S3ObjectVersionT&& value) {
    m_s3ObjectVersionHasBeenSet = true;
    m_s3ObjectVersion = std::forward<S3ObjectVersionT>(value);
  }
  template <typename S3ObjectVersionT = Aws::String>
  LayerVersionContentInput& WithS3ObjectVersion(S3ObjectVersionT&& value) {
    SetS3ObjectVersion(std::forward<S3ObjectVersionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The base64-encoded contents of the layer archive. Amazon Web Services SDK and
   * Amazon Web Services CLI clients handle the encoding for you.</p>
   */
  inline const Aws::Utils::CryptoBuffer& GetZipFile() const { return m_zipFile; }
  inline bool ZipFileHasBeenSet() const { return m_zipFileHasBeenSet; }
  template <typename ZipFileT = Aws::Utils::CryptoBuffer>
  void SetZipFile(ZipFileT&& value) {
    m_zipFileHasBeenSet = true;
    m_zipFile = std::forward<ZipFileT>(value);
  }
  template <typename ZipFileT = Aws::Utils::CryptoBuffer>
  LayerVersionContentInput& WithZipFile(ZipFileT&& value) {
    SetZipFile(std::forward<ZipFileT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_s3Bucket;
  bool m_s3BucketHasBeenSet = false;

  Aws::String m_s3Key;
  bool m_s3KeyHasBeenSet = false;

  Aws::String m_s3ObjectVersion;
  bool m_s3ObjectVersionHasBeenSet = false;

  Aws::Utils::CryptoBuffer m_zipFile{};
  bool m_zipFileHasBeenSet = false;
};

}  // namespace Model
}  // namespace Lambda
}  // namespace Aws
