﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/UUID.h>
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/medialive/MediaLiveRequest.h>
#include <aws/medialive/MediaLive_EXPORTS.h>
#include <aws/medialive/model/ClusterNetworkSettingsCreateRequest.h>
#include <aws/medialive/model/ClusterType.h>

#include <utility>

namespace Aws {
namespace MediaLive {
namespace Model {

/**
 * Create as many Clusters as you want, but create at least one. Each Cluster
 * groups together Nodes that you want to treat as a collection. Within the
 * Cluster, you will set up some Nodes as active Nodes, and some as backup Nodes,
 * for Node failover purposes. Each Node can belong to only one Cluster.<p><h3>See
 * Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/medialive-2017-10-14/CreateClusterRequest">AWS
 * API Reference</a></p>
 */
class CreateClusterRequest : public MediaLiveRequest {
 public:
  AWS_MEDIALIVE_API CreateClusterRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateCluster"; }

  AWS_MEDIALIVE_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * Specify a type. All the Nodes that you later add to this Cluster must be this
   * type of hardware. One Cluster instance can't contain different hardware types.
   * You won't be able to change this parameter after you create the Cluster.
   */
  inline ClusterType GetClusterType() const { return m_clusterType; }
  inline bool ClusterTypeHasBeenSet() const { return m_clusterTypeHasBeenSet; }
  inline void SetClusterType(ClusterType value) {
    m_clusterTypeHasBeenSet = true;
    m_clusterType = value;
  }
  inline CreateClusterRequest& WithClusterType(ClusterType value) {
    SetClusterType(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * The ARN of the IAM role for the Node in this Cluster. The role must include all
   * the operations that you expect these Node to perform. If necessary, create a
   * role in IAM, then attach it here.
   */
  inline const Aws::String& GetInstanceRoleArn() const { return m_instanceRoleArn; }
  inline bool InstanceRoleArnHasBeenSet() const { return m_instanceRoleArnHasBeenSet; }
  template <typename InstanceRoleArnT = Aws::String>
  void SetInstanceRoleArn(InstanceRoleArnT&& value) {
    m_instanceRoleArnHasBeenSet = true;
    m_instanceRoleArn = std::forward<InstanceRoleArnT>(value);
  }
  template <typename InstanceRoleArnT = Aws::String>
  CreateClusterRequest& WithInstanceRoleArn(InstanceRoleArnT&& value) {
    SetInstanceRoleArn(std::forward<InstanceRoleArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * Specify a name that is unique in the AWS account. We recommend that you assign a
   * name that hints at the types of Nodes in the Cluster. Names are case-sensitive.
   */
  inline const Aws::String& GetName() const { return m_name; }
  inline bool NameHasBeenSet() const { return m_nameHasBeenSet; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  CreateClusterRequest& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * Network settings that connect the Nodes in the Cluster to one or more of the
   * Networks that the Cluster is associated with.
   */
  inline const ClusterNetworkSettingsCreateRequest& GetNetworkSettings() const { return m_networkSettings; }
  inline bool NetworkSettingsHasBeenSet() const { return m_networkSettingsHasBeenSet; }
  template <typename NetworkSettingsT = ClusterNetworkSettingsCreateRequest>
  void SetNetworkSettings(NetworkSettingsT&& value) {
    m_networkSettingsHasBeenSet = true;
    m_networkSettings = std::forward<NetworkSettingsT>(value);
  }
  template <typename NetworkSettingsT = ClusterNetworkSettingsCreateRequest>
  CreateClusterRequest& WithNetworkSettings(NetworkSettingsT&& value) {
    SetNetworkSettings(std::forward<NetworkSettingsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * The unique ID of the request.
   */
  inline const Aws::String& GetRequestId() const { return m_requestId; }
  inline bool RequestIdHasBeenSet() const { return m_requestIdHasBeenSet; }
  template <typename RequestIdT = Aws::String>
  void SetRequestId(RequestIdT&& value) {
    m_requestIdHasBeenSet = true;
    m_requestId = std::forward<RequestIdT>(value);
  }
  template <typename RequestIdT = Aws::String>
  CreateClusterRequest& WithRequestId(RequestIdT&& value) {
    SetRequestId(std::forward<RequestIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * A collection of key-value pairs.
   */
  inline const Aws::Map<Aws::String, Aws::String>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  CreateClusterRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsKeyT = Aws::String, typename TagsValueT = Aws::String>
  CreateClusterRequest& AddTags(TagsKeyT&& key, TagsValueT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace(std::forward<TagsKeyT>(key), std::forward<TagsValueT>(value));
    return *this;
  }
  ///@}
 private:
  ClusterType m_clusterType{ClusterType::NOT_SET};
  bool m_clusterTypeHasBeenSet = false;

  Aws::String m_instanceRoleArn;
  bool m_instanceRoleArnHasBeenSet = false;

  Aws::String m_name;
  bool m_nameHasBeenSet = false;

  ClusterNetworkSettingsCreateRequest m_networkSettings;
  bool m_networkSettingsHasBeenSet = false;

  Aws::String m_requestId{Aws::Utils::UUID::PseudoRandomUUID()};
  bool m_requestIdHasBeenSet = true;

  Aws::Map<Aws::String, Aws::String> m_tags;
  bool m_tagsHasBeenSet = false;
};

}  // namespace Model
}  // namespace MediaLive
}  // namespace Aws
