﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/DateTime.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/personalize/Personalize_EXPORTS.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace Personalize {
namespace Model {

/**
 * <p>Provides information about a recipe. Each recipe provides an algorithm that
 * Amazon Personalize uses in model training when you use the <a
 * href="https://docs.aws.amazon.com/personalize/latest/dg/API_CreateSolution.html">CreateSolution</a>
 * operation. </p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/personalize-2018-05-22/Recipe">AWS
 * API Reference</a></p>
 */
class Recipe {
 public:
  AWS_PERSONALIZE_API Recipe() = default;
  AWS_PERSONALIZE_API Recipe(Aws::Utils::Json::JsonView jsonValue);
  AWS_PERSONALIZE_API Recipe& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_PERSONALIZE_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>The name of the recipe.</p>
   */
  inline const Aws::String& GetName() const { return m_name; }
  inline bool NameHasBeenSet() const { return m_nameHasBeenSet; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  Recipe& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the recipe.</p>
   */
  inline const Aws::String& GetRecipeArn() const { return m_recipeArn; }
  inline bool RecipeArnHasBeenSet() const { return m_recipeArnHasBeenSet; }
  template <typename RecipeArnT = Aws::String>
  void SetRecipeArn(RecipeArnT&& value) {
    m_recipeArnHasBeenSet = true;
    m_recipeArn = std::forward<RecipeArnT>(value);
  }
  template <typename RecipeArnT = Aws::String>
  Recipe& WithRecipeArn(RecipeArnT&& value) {
    SetRecipeArn(std::forward<RecipeArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the algorithm that Amazon Personalize uses
   * to train the model.</p>
   */
  inline const Aws::String& GetAlgorithmArn() const { return m_algorithmArn; }
  inline bool AlgorithmArnHasBeenSet() const { return m_algorithmArnHasBeenSet; }
  template <typename AlgorithmArnT = Aws::String>
  void SetAlgorithmArn(AlgorithmArnT&& value) {
    m_algorithmArnHasBeenSet = true;
    m_algorithmArn = std::forward<AlgorithmArnT>(value);
  }
  template <typename AlgorithmArnT = Aws::String>
  Recipe& WithAlgorithmArn(AlgorithmArnT&& value) {
    SetAlgorithmArn(std::forward<AlgorithmArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ARN of the FeatureTransformation object.</p>
   */
  inline const Aws::String& GetFeatureTransformationArn() const { return m_featureTransformationArn; }
  inline bool FeatureTransformationArnHasBeenSet() const { return m_featureTransformationArnHasBeenSet; }
  template <typename FeatureTransformationArnT = Aws::String>
  void SetFeatureTransformationArn(FeatureTransformationArnT&& value) {
    m_featureTransformationArnHasBeenSet = true;
    m_featureTransformationArn = std::forward<FeatureTransformationArnT>(value);
  }
  template <typename FeatureTransformationArnT = Aws::String>
  Recipe& WithFeatureTransformationArn(FeatureTransformationArnT&& value) {
    SetFeatureTransformationArn(std::forward<FeatureTransformationArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The status of the recipe.</p>
   */
  inline const Aws::String& GetStatus() const { return m_status; }
  inline bool StatusHasBeenSet() const { return m_statusHasBeenSet; }
  template <typename StatusT = Aws::String>
  void SetStatus(StatusT&& value) {
    m_statusHasBeenSet = true;
    m_status = std::forward<StatusT>(value);
  }
  template <typename StatusT = Aws::String>
  Recipe& WithStatus(StatusT&& value) {
    SetStatus(std::forward<StatusT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The description of the recipe.</p>
   */
  inline const Aws::String& GetDescription() const { return m_description; }
  inline bool DescriptionHasBeenSet() const { return m_descriptionHasBeenSet; }
  template <typename DescriptionT = Aws::String>
  void SetDescription(DescriptionT&& value) {
    m_descriptionHasBeenSet = true;
    m_description = std::forward<DescriptionT>(value);
  }
  template <typename DescriptionT = Aws::String>
  Recipe& WithDescription(DescriptionT&& value) {
    SetDescription(std::forward<DescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The date and time (in Unix format) that the recipe was created.</p>
   */
  inline const Aws::Utils::DateTime& GetCreationDateTime() const { return m_creationDateTime; }
  inline bool CreationDateTimeHasBeenSet() const { return m_creationDateTimeHasBeenSet; }
  template <typename CreationDateTimeT = Aws::Utils::DateTime>
  void SetCreationDateTime(CreationDateTimeT&& value) {
    m_creationDateTimeHasBeenSet = true;
    m_creationDateTime = std::forward<CreationDateTimeT>(value);
  }
  template <typename CreationDateTimeT = Aws::Utils::DateTime>
  Recipe& WithCreationDateTime(CreationDateTimeT&& value) {
    SetCreationDateTime(std::forward<CreationDateTimeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>One of the following values:</p> <ul> <li> <p>PERSONALIZED_RANKING</p> </li>
   * <li> <p>RELATED_ITEMS</p> </li> <li> <p>USER_PERSONALIZATION</p> </li> </ul>
   */
  inline const Aws::String& GetRecipeType() const { return m_recipeType; }
  inline bool RecipeTypeHasBeenSet() const { return m_recipeTypeHasBeenSet; }
  template <typename RecipeTypeT = Aws::String>
  void SetRecipeType(RecipeTypeT&& value) {
    m_recipeTypeHasBeenSet = true;
    m_recipeType = std::forward<RecipeTypeT>(value);
  }
  template <typename RecipeTypeT = Aws::String>
  Recipe& WithRecipeType(RecipeTypeT&& value) {
    SetRecipeType(std::forward<RecipeTypeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The date and time (in Unix format) that the recipe was last updated.</p>
   */
  inline const Aws::Utils::DateTime& GetLastUpdatedDateTime() const { return m_lastUpdatedDateTime; }
  inline bool LastUpdatedDateTimeHasBeenSet() const { return m_lastUpdatedDateTimeHasBeenSet; }
  template <typename LastUpdatedDateTimeT = Aws::Utils::DateTime>
  void SetLastUpdatedDateTime(LastUpdatedDateTimeT&& value) {
    m_lastUpdatedDateTimeHasBeenSet = true;
    m_lastUpdatedDateTime = std::forward<LastUpdatedDateTimeT>(value);
  }
  template <typename LastUpdatedDateTimeT = Aws::Utils::DateTime>
  Recipe& WithLastUpdatedDateTime(LastUpdatedDateTimeT&& value) {
    SetLastUpdatedDateTime(std::forward<LastUpdatedDateTimeT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_name;
  bool m_nameHasBeenSet = false;

  Aws::String m_recipeArn;
  bool m_recipeArnHasBeenSet = false;

  Aws::String m_algorithmArn;
  bool m_algorithmArnHasBeenSet = false;

  Aws::String m_featureTransformationArn;
  bool m_featureTransformationArnHasBeenSet = false;

  Aws::String m_status;
  bool m_statusHasBeenSet = false;

  Aws::String m_description;
  bool m_descriptionHasBeenSet = false;

  Aws::Utils::DateTime m_creationDateTime{};
  bool m_creationDateTimeHasBeenSet = false;

  Aws::String m_recipeType;
  bool m_recipeTypeHasBeenSet = false;

  Aws::Utils::DateTime m_lastUpdatedDateTime{};
  bool m_lastUpdatedDateTimeHasBeenSet = false;
};

}  // namespace Model
}  // namespace Personalize
}  // namespace Aws
