﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/ssm-contacts/SSMContactsRequest.h>
#include <aws/ssm-contacts/SSMContacts_EXPORTS.h>
#include <aws/ssm-contacts/model/Plan.h>

#include <utility>

namespace Aws {
namespace SSMContacts {
namespace Model {

/**
 */
class UpdateContactRequest : public SSMContactsRequest {
 public:
  AWS_SSMCONTACTS_API UpdateContactRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "UpdateContact"; }

  AWS_SSMCONTACTS_API Aws::String SerializePayload() const override;

  AWS_SSMCONTACTS_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the contact or escalation plan you're
   * updating.</p>
   */
  inline const Aws::String& GetContactId() const { return m_contactId; }
  inline bool ContactIdHasBeenSet() const { return m_contactIdHasBeenSet; }
  template <typename ContactIdT = Aws::String>
  void SetContactId(ContactIdT&& value) {
    m_contactIdHasBeenSet = true;
    m_contactId = std::forward<ContactIdT>(value);
  }
  template <typename ContactIdT = Aws::String>
  UpdateContactRequest& WithContactId(ContactIdT&& value) {
    SetContactId(std::forward<ContactIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The full name of the contact or escalation plan.</p>
   */
  inline const Aws::String& GetDisplayName() const { return m_displayName; }
  inline bool DisplayNameHasBeenSet() const { return m_displayNameHasBeenSet; }
  template <typename DisplayNameT = Aws::String>
  void SetDisplayName(DisplayNameT&& value) {
    m_displayNameHasBeenSet = true;
    m_displayName = std::forward<DisplayNameT>(value);
  }
  template <typename DisplayNameT = Aws::String>
  UpdateContactRequest& WithDisplayName(DisplayNameT&& value) {
    SetDisplayName(std::forward<DisplayNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A list of stages. A contact has an engagement plan with stages for specified
   * contact channels. An escalation plan uses these stages to contact specified
   * contacts.</p>
   */
  inline const Plan& GetPlan() const { return m_plan; }
  inline bool PlanHasBeenSet() const { return m_planHasBeenSet; }
  template <typename PlanT = Plan>
  void SetPlan(PlanT&& value) {
    m_planHasBeenSet = true;
    m_plan = std::forward<PlanT>(value);
  }
  template <typename PlanT = Plan>
  UpdateContactRequest& WithPlan(PlanT&& value) {
    SetPlan(std::forward<PlanT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_contactId;
  bool m_contactIdHasBeenSet = false;

  Aws::String m_displayName;
  bool m_displayNameHasBeenSet = false;

  Plan m_plan;
  bool m_planHasBeenSet = false;
};

}  // namespace Model
}  // namespace SSMContacts
}  // namespace Aws
