﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSStreamFwd.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/ec2/EC2_EXPORTS.h>
#include <aws/ec2/model/InstanceRequirements.h>
#include <aws/ec2/model/InstanceType.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Xml {
class XmlNode;
}  // namespace Xml
}  // namespace Utils
namespace EC2 {
namespace Model {

/**
 * <p>Describes overrides for a launch template.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/LaunchTemplateOverrides">AWS
 * API Reference</a></p>
 */
class LaunchTemplateOverrides {
 public:
  AWS_EC2_API LaunchTemplateOverrides() = default;
  AWS_EC2_API LaunchTemplateOverrides(const Aws::Utils::Xml::XmlNode& xmlNode);
  AWS_EC2_API LaunchTemplateOverrides& operator=(const Aws::Utils::Xml::XmlNode& xmlNode);

  AWS_EC2_API void OutputToStream(Aws::OStream& ostream, const char* location, unsigned index, const char* locationValue) const;
  AWS_EC2_API void OutputToStream(Aws::OStream& oStream, const char* location) const;

  ///@{
  /**
   * <p>The instance type.</p>
   */
  inline InstanceType GetInstanceType() const { return m_instanceType; }
  inline bool InstanceTypeHasBeenSet() const { return m_instanceTypeHasBeenSet; }
  inline void SetInstanceType(InstanceType value) {
    m_instanceTypeHasBeenSet = true;
    m_instanceType = value;
  }
  inline LaunchTemplateOverrides& WithInstanceType(InstanceType value) {
    SetInstanceType(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum price per unit hour that you are willing to pay for a Spot
   * Instance. We do not recommend using this parameter because it can lead to
   * increased interruptions. If you do not specify this parameter, you will pay the
   * current Spot price.</p>  <p>If you specify a maximum price, your
   * instances will be interrupted more frequently than if you do not specify this
   * parameter.</p>
   */
  inline const Aws::String& GetSpotPrice() const { return m_spotPrice; }
  inline bool SpotPriceHasBeenSet() const { return m_spotPriceHasBeenSet; }
  template <typename SpotPriceT = Aws::String>
  void SetSpotPrice(SpotPriceT&& value) {
    m_spotPriceHasBeenSet = true;
    m_spotPrice = std::forward<SpotPriceT>(value);
  }
  template <typename SpotPriceT = Aws::String>
  LaunchTemplateOverrides& WithSpotPrice(SpotPriceT&& value) {
    SetSpotPrice(std::forward<SpotPriceT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of the subnet in which to launch the instances.</p>
   */
  inline const Aws::String& GetSubnetId() const { return m_subnetId; }
  inline bool SubnetIdHasBeenSet() const { return m_subnetIdHasBeenSet; }
  template <typename SubnetIdT = Aws::String>
  void SetSubnetId(SubnetIdT&& value) {
    m_subnetIdHasBeenSet = true;
    m_subnetId = std::forward<SubnetIdT>(value);
  }
  template <typename SubnetIdT = Aws::String>
  LaunchTemplateOverrides& WithSubnetId(SubnetIdT&& value) {
    SetSubnetId(std::forward<SubnetIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Availability Zone in which to launch the instances.</p>
   */
  inline const Aws::String& GetAvailabilityZone() const { return m_availabilityZone; }
  inline bool AvailabilityZoneHasBeenSet() const { return m_availabilityZoneHasBeenSet; }
  template <typename AvailabilityZoneT = Aws::String>
  void SetAvailabilityZone(AvailabilityZoneT&& value) {
    m_availabilityZoneHasBeenSet = true;
    m_availabilityZone = std::forward<AvailabilityZoneT>(value);
  }
  template <typename AvailabilityZoneT = Aws::String>
  LaunchTemplateOverrides& WithAvailabilityZone(AvailabilityZoneT&& value) {
    SetAvailabilityZone(std::forward<AvailabilityZoneT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The number of units provided by the specified instance type. These are the
   * same units that you chose to set the target capacity in terms of instances, or a
   * performance characteristic such as vCPUs, memory, or I/O.</p> <p>If the target
   * capacity divided by this value is not a whole number, Amazon EC2 rounds the
   * number of instances to the next whole number. If this value is not specified,
   * the default is 1.</p>  <p>When specifying weights, the price used in the
   * <code>lowestPrice</code> and <code>priceCapacityOptimized</code> allocation
   * strategies is per <i>unit</i> hour (where the instance price is divided by the
   * specified weight). However, if all the specified weights are above the requested
   * <code>TargetCapacity</code>, resulting in only 1 instance being launched, the
   * price used is per <i>instance</i> hour.</p>
   */
  inline double GetWeightedCapacity() const { return m_weightedCapacity; }
  inline bool WeightedCapacityHasBeenSet() const { return m_weightedCapacityHasBeenSet; }
  inline void SetWeightedCapacity(double value) {
    m_weightedCapacityHasBeenSet = true;
    m_weightedCapacity = value;
  }
  inline LaunchTemplateOverrides& WithWeightedCapacity(double value) {
    SetWeightedCapacity(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The priority for the launch template override. The highest priority is
   * launched first.</p> <p>If <code>OnDemandAllocationStrategy</code> is set to
   * <code>prioritized</code>, Spot Fleet uses priority to determine which launch
   * template override to use first in fulfilling On-Demand capacity.</p> <p>If the
   * Spot <code>AllocationStrategy</code> is set to
   * <code>capacityOptimizedPrioritized</code>, Spot Fleet uses priority on a
   * best-effort basis to determine which launch template override to use in
   * fulfilling Spot capacity, but optimizes for capacity first.</p> <p>Valid values
   * are whole numbers starting at <code>0</code>. The lower the number, the higher
   * the priority. If no number is set, the launch template override has the lowest
   * priority. You can set the same priority for different launch template
   * overrides.</p>
   */
  inline double GetPriority() const { return m_priority; }
  inline bool PriorityHasBeenSet() const { return m_priorityHasBeenSet; }
  inline void SetPriority(double value) {
    m_priorityHasBeenSet = true;
    m_priority = value;
  }
  inline LaunchTemplateOverrides& WithPriority(double value) {
    SetPriority(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The instance requirements. When you specify instance requirements, Amazon EC2
   * will identify instance types with the provided requirements, and then use your
   * On-Demand and Spot allocation strategies to launch instances from these instance
   * types, in the same way as when you specify a list of instance types.</p>
   * <p>If you specify <code>InstanceRequirements</code>, you can't specify
   * <code>InstanceType</code>.</p>
   */
  inline const InstanceRequirements& GetInstanceRequirements() const { return m_instanceRequirements; }
  inline bool InstanceRequirementsHasBeenSet() const { return m_instanceRequirementsHasBeenSet; }
  template <typename InstanceRequirementsT = InstanceRequirements>
  void SetInstanceRequirements(InstanceRequirementsT&& value) {
    m_instanceRequirementsHasBeenSet = true;
    m_instanceRequirements = std::forward<InstanceRequirementsT>(value);
  }
  template <typename InstanceRequirementsT = InstanceRequirements>
  LaunchTemplateOverrides& WithInstanceRequirements(InstanceRequirementsT&& value) {
    SetInstanceRequirements(std::forward<InstanceRequirementsT>(value));
    return *this;
  }
  ///@}
 private:
  InstanceType m_instanceType{InstanceType::NOT_SET};
  bool m_instanceTypeHasBeenSet = false;

  Aws::String m_spotPrice;
  bool m_spotPriceHasBeenSet = false;

  Aws::String m_subnetId;
  bool m_subnetIdHasBeenSet = false;

  Aws::String m_availabilityZone;
  bool m_availabilityZoneHasBeenSet = false;

  double m_weightedCapacity{0.0};
  bool m_weightedCapacityHasBeenSet = false;

  double m_priority{0.0};
  bool m_priorityHasBeenSet = false;

  InstanceRequirements m_instanceRequirements;
  bool m_instanceRequirementsHasBeenSet = false;
};

}  // namespace Model
}  // namespace EC2
}  // namespace Aws
