﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/DateTime.h>
#include <aws/core/utils/memory/stl/AWSStreamFwd.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/ec2/EC2_EXPORTS.h>
#include <aws/ec2/model/ImageMetadata.h>
#include <aws/ec2/model/InstanceState.h>
#include <aws/ec2/model/InstanceType.h>
#include <aws/ec2/model/OperatorResponse.h>
#include <aws/ec2/model/Tag.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Xml {
class XmlNode;
}  // namespace Xml
}  // namespace Utils
namespace EC2 {
namespace Model {

/**
 * <p>Information about the instance and the AMI used to launch the
 * instance.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/InstanceImageMetadata">AWS
 * API Reference</a></p>
 */
class InstanceImageMetadata {
 public:
  AWS_EC2_API InstanceImageMetadata() = default;
  AWS_EC2_API InstanceImageMetadata(const Aws::Utils::Xml::XmlNode& xmlNode);
  AWS_EC2_API InstanceImageMetadata& operator=(const Aws::Utils::Xml::XmlNode& xmlNode);

  AWS_EC2_API void OutputToStream(Aws::OStream& ostream, const char* location, unsigned index, const char* locationValue) const;
  AWS_EC2_API void OutputToStream(Aws::OStream& oStream, const char* location) const;

  ///@{
  /**
   * <p>The ID of the instance.</p>
   */
  inline const Aws::String& GetInstanceId() const { return m_instanceId; }
  inline bool InstanceIdHasBeenSet() const { return m_instanceIdHasBeenSet; }
  template <typename InstanceIdT = Aws::String>
  void SetInstanceId(InstanceIdT&& value) {
    m_instanceIdHasBeenSet = true;
    m_instanceId = std::forward<InstanceIdT>(value);
  }
  template <typename InstanceIdT = Aws::String>
  InstanceImageMetadata& WithInstanceId(InstanceIdT&& value) {
    SetInstanceId(std::forward<InstanceIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The instance type.</p>
   */
  inline InstanceType GetInstanceType() const { return m_instanceType; }
  inline bool InstanceTypeHasBeenSet() const { return m_instanceTypeHasBeenSet; }
  inline void SetInstanceType(InstanceType value) {
    m_instanceTypeHasBeenSet = true;
    m_instanceType = value;
  }
  inline InstanceImageMetadata& WithInstanceType(InstanceType value) {
    SetInstanceType(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The time the instance was launched.</p>
   */
  inline const Aws::Utils::DateTime& GetLaunchTime() const { return m_launchTime; }
  inline bool LaunchTimeHasBeenSet() const { return m_launchTimeHasBeenSet; }
  template <typename LaunchTimeT = Aws::Utils::DateTime>
  void SetLaunchTime(LaunchTimeT&& value) {
    m_launchTimeHasBeenSet = true;
    m_launchTime = std::forward<LaunchTimeT>(value);
  }
  template <typename LaunchTimeT = Aws::Utils::DateTime>
  InstanceImageMetadata& WithLaunchTime(LaunchTimeT&& value) {
    SetLaunchTime(std::forward<LaunchTimeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Availability Zone or Local Zone of the instance.</p>
   */
  inline const Aws::String& GetAvailabilityZone() const { return m_availabilityZone; }
  inline bool AvailabilityZoneHasBeenSet() const { return m_availabilityZoneHasBeenSet; }
  template <typename AvailabilityZoneT = Aws::String>
  void SetAvailabilityZone(AvailabilityZoneT&& value) {
    m_availabilityZoneHasBeenSet = true;
    m_availabilityZone = std::forward<AvailabilityZoneT>(value);
  }
  template <typename AvailabilityZoneT = Aws::String>
  InstanceImageMetadata& WithAvailabilityZone(AvailabilityZoneT&& value) {
    SetAvailabilityZone(std::forward<AvailabilityZoneT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of the Availability Zone or Local Zone of the instance.</p>
   */
  inline const Aws::String& GetZoneId() const { return m_zoneId; }
  inline bool ZoneIdHasBeenSet() const { return m_zoneIdHasBeenSet; }
  template <typename ZoneIdT = Aws::String>
  void SetZoneId(ZoneIdT&& value) {
    m_zoneIdHasBeenSet = true;
    m_zoneId = std::forward<ZoneIdT>(value);
  }
  template <typename ZoneIdT = Aws::String>
  InstanceImageMetadata& WithZoneId(ZoneIdT&& value) {
    SetZoneId(std::forward<ZoneIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The current state of the instance.</p>
   */
  inline const InstanceState& GetState() const { return m_state; }
  inline bool StateHasBeenSet() const { return m_stateHasBeenSet; }
  template <typename StateT = InstanceState>
  void SetState(StateT&& value) {
    m_stateHasBeenSet = true;
    m_state = std::forward<StateT>(value);
  }
  template <typename StateT = InstanceState>
  InstanceImageMetadata& WithState(StateT&& value) {
    SetState(std::forward<StateT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of the Amazon Web Services account that owns the instance.</p>
   */
  inline const Aws::String& GetOwnerId() const { return m_ownerId; }
  inline bool OwnerIdHasBeenSet() const { return m_ownerIdHasBeenSet; }
  template <typename OwnerIdT = Aws::String>
  void SetOwnerId(OwnerIdT&& value) {
    m_ownerIdHasBeenSet = true;
    m_ownerId = std::forward<OwnerIdT>(value);
  }
  template <typename OwnerIdT = Aws::String>
  InstanceImageMetadata& WithOwnerId(OwnerIdT&& value) {
    SetOwnerId(std::forward<OwnerIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Any tags assigned to the instance.</p>
   */
  inline const Aws::Vector<Tag>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Vector<Tag>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Vector<Tag>>
  InstanceImageMetadata& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsT = Tag>
  InstanceImageMetadata& AddTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace_back(std::forward<TagsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Information about the AMI used to launch the instance.</p>
   */
  inline const ImageMetadata& GetImageMetadata() const { return m_imageMetadata; }
  inline bool ImageMetadataHasBeenSet() const { return m_imageMetadataHasBeenSet; }
  template <typename ImageMetadataT = ImageMetadata>
  void SetImageMetadata(ImageMetadataT&& value) {
    m_imageMetadataHasBeenSet = true;
    m_imageMetadata = std::forward<ImageMetadataT>(value);
  }
  template <typename ImageMetadataT = ImageMetadata>
  InstanceImageMetadata& WithImageMetadata(ImageMetadataT&& value) {
    SetImageMetadata(std::forward<ImageMetadataT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The entity that manages the instance.</p>
   */
  inline const OperatorResponse& GetOperator() const { return m_operator; }
  inline bool OperatorHasBeenSet() const { return m_operatorHasBeenSet; }
  template <typename OperatorT = OperatorResponse>
  void SetOperator(OperatorT&& value) {
    m_operatorHasBeenSet = true;
    m_operator = std::forward<OperatorT>(value);
  }
  template <typename OperatorT = OperatorResponse>
  InstanceImageMetadata& WithOperator(OperatorT&& value) {
    SetOperator(std::forward<OperatorT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_instanceId;
  bool m_instanceIdHasBeenSet = false;

  InstanceType m_instanceType{InstanceType::NOT_SET};
  bool m_instanceTypeHasBeenSet = false;

  Aws::Utils::DateTime m_launchTime{};
  bool m_launchTimeHasBeenSet = false;

  Aws::String m_availabilityZone;
  bool m_availabilityZoneHasBeenSet = false;

  Aws::String m_zoneId;
  bool m_zoneIdHasBeenSet = false;

  InstanceState m_state;
  bool m_stateHasBeenSet = false;

  Aws::String m_ownerId;
  bool m_ownerIdHasBeenSet = false;

  Aws::Vector<Tag> m_tags;
  bool m_tagsHasBeenSet = false;

  ImageMetadata m_imageMetadata;
  bool m_imageMetadataHasBeenSet = false;

  OperatorResponse m_operator;
  bool m_operatorHasBeenSet = false;
};

}  // namespace Model
}  // namespace EC2
}  // namespace Aws
