﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSStreamFwd.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/ec2/EC2_EXPORTS.h>
#include <aws/ec2/model/CurrencyCodeValues.h>
#include <aws/ec2/model/PaymentOption.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Xml {
class XmlNode;
}  // namespace Xml
}  // namespace Utils
namespace EC2 {
namespace Model {

/**
 * <p>Details about the Dedicated Host Reservation offering.</p><p><h3>See
 * Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/HostOffering">AWS
 * API Reference</a></p>
 */
class HostOffering {
 public:
  AWS_EC2_API HostOffering() = default;
  AWS_EC2_API HostOffering(const Aws::Utils::Xml::XmlNode& xmlNode);
  AWS_EC2_API HostOffering& operator=(const Aws::Utils::Xml::XmlNode& xmlNode);

  AWS_EC2_API void OutputToStream(Aws::OStream& ostream, const char* location, unsigned index, const char* locationValue) const;
  AWS_EC2_API void OutputToStream(Aws::OStream& oStream, const char* location) const;

  ///@{
  /**
   * <p>The currency of the offering.</p>
   */
  inline CurrencyCodeValues GetCurrencyCode() const { return m_currencyCode; }
  inline bool CurrencyCodeHasBeenSet() const { return m_currencyCodeHasBeenSet; }
  inline void SetCurrencyCode(CurrencyCodeValues value) {
    m_currencyCodeHasBeenSet = true;
    m_currencyCode = value;
  }
  inline HostOffering& WithCurrencyCode(CurrencyCodeValues value) {
    SetCurrencyCode(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The duration of the offering (in seconds).</p>
   */
  inline int GetDuration() const { return m_duration; }
  inline bool DurationHasBeenSet() const { return m_durationHasBeenSet; }
  inline void SetDuration(int value) {
    m_durationHasBeenSet = true;
    m_duration = value;
  }
  inline HostOffering& WithDuration(int value) {
    SetDuration(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The hourly price of the offering.</p>
   */
  inline const Aws::String& GetHourlyPrice() const { return m_hourlyPrice; }
  inline bool HourlyPriceHasBeenSet() const { return m_hourlyPriceHasBeenSet; }
  template <typename HourlyPriceT = Aws::String>
  void SetHourlyPrice(HourlyPriceT&& value) {
    m_hourlyPriceHasBeenSet = true;
    m_hourlyPrice = std::forward<HourlyPriceT>(value);
  }
  template <typename HourlyPriceT = Aws::String>
  HostOffering& WithHourlyPrice(HourlyPriceT&& value) {
    SetHourlyPrice(std::forward<HourlyPriceT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The instance family of the offering.</p>
   */
  inline const Aws::String& GetInstanceFamily() const { return m_instanceFamily; }
  inline bool InstanceFamilyHasBeenSet() const { return m_instanceFamilyHasBeenSet; }
  template <typename InstanceFamilyT = Aws::String>
  void SetInstanceFamily(InstanceFamilyT&& value) {
    m_instanceFamilyHasBeenSet = true;
    m_instanceFamily = std::forward<InstanceFamilyT>(value);
  }
  template <typename InstanceFamilyT = Aws::String>
  HostOffering& WithInstanceFamily(InstanceFamilyT&& value) {
    SetInstanceFamily(std::forward<InstanceFamilyT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of the offering.</p>
   */
  inline const Aws::String& GetOfferingId() const { return m_offeringId; }
  inline bool OfferingIdHasBeenSet() const { return m_offeringIdHasBeenSet; }
  template <typename OfferingIdT = Aws::String>
  void SetOfferingId(OfferingIdT&& value) {
    m_offeringIdHasBeenSet = true;
    m_offeringId = std::forward<OfferingIdT>(value);
  }
  template <typename OfferingIdT = Aws::String>
  HostOffering& WithOfferingId(OfferingIdT&& value) {
    SetOfferingId(std::forward<OfferingIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The available payment option.</p>
   */
  inline PaymentOption GetPaymentOption() const { return m_paymentOption; }
  inline bool PaymentOptionHasBeenSet() const { return m_paymentOptionHasBeenSet; }
  inline void SetPaymentOption(PaymentOption value) {
    m_paymentOptionHasBeenSet = true;
    m_paymentOption = value;
  }
  inline HostOffering& WithPaymentOption(PaymentOption value) {
    SetPaymentOption(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The upfront price of the offering. Does not apply to No Upfront
   * offerings.</p>
   */
  inline const Aws::String& GetUpfrontPrice() const { return m_upfrontPrice; }
  inline bool UpfrontPriceHasBeenSet() const { return m_upfrontPriceHasBeenSet; }
  template <typename UpfrontPriceT = Aws::String>
  void SetUpfrontPrice(UpfrontPriceT&& value) {
    m_upfrontPriceHasBeenSet = true;
    m_upfrontPrice = std::forward<UpfrontPriceT>(value);
  }
  template <typename UpfrontPriceT = Aws::String>
  HostOffering& WithUpfrontPrice(UpfrontPriceT&& value) {
    SetUpfrontPrice(std::forward<UpfrontPriceT>(value));
    return *this;
  }
  ///@}
 private:
  CurrencyCodeValues m_currencyCode{CurrencyCodeValues::NOT_SET};
  bool m_currencyCodeHasBeenSet = false;

  int m_duration{0};
  bool m_durationHasBeenSet = false;

  Aws::String m_hourlyPrice;
  bool m_hourlyPriceHasBeenSet = false;

  Aws::String m_instanceFamily;
  bool m_instanceFamilyHasBeenSet = false;

  Aws::String m_offeringId;
  bool m_offeringIdHasBeenSet = false;

  PaymentOption m_paymentOption{PaymentOption::NOT_SET};
  bool m_paymentOptionHasBeenSet = false;

  Aws::String m_upfrontPrice;
  bool m_upfrontPriceHasBeenSet = false;
};

}  // namespace Model
}  // namespace EC2
}  // namespace Aws
