﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/ec2/EC2Request.h>
#include <aws/ec2/EC2_EXPORTS.h>
#include <aws/ec2/model/Filter.h>

#include <utility>

namespace Aws {
namespace EC2 {
namespace Model {

/**
 */
class DescribeVerifiedAccessEndpointsRequest : public EC2Request {
 public:
  AWS_EC2_API DescribeVerifiedAccessEndpointsRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "DescribeVerifiedAccessEndpoints"; }

  AWS_EC2_API Aws::String SerializePayload() const override;

 protected:
  AWS_EC2_API void DumpBodyToUrl(Aws::Http::URI& uri) const override;

 public:
  ///@{
  /**
   * <p>The ID of the Verified Access endpoint.</p>
   */
  inline const Aws::Vector<Aws::String>& GetVerifiedAccessEndpointIds() const { return m_verifiedAccessEndpointIds; }
  inline bool VerifiedAccessEndpointIdsHasBeenSet() const { return m_verifiedAccessEndpointIdsHasBeenSet; }
  template <typename VerifiedAccessEndpointIdsT = Aws::Vector<Aws::String>>
  void SetVerifiedAccessEndpointIds(VerifiedAccessEndpointIdsT&& value) {
    m_verifiedAccessEndpointIdsHasBeenSet = true;
    m_verifiedAccessEndpointIds = std::forward<VerifiedAccessEndpointIdsT>(value);
  }
  template <typename VerifiedAccessEndpointIdsT = Aws::Vector<Aws::String>>
  DescribeVerifiedAccessEndpointsRequest& WithVerifiedAccessEndpointIds(VerifiedAccessEndpointIdsT&& value) {
    SetVerifiedAccessEndpointIds(std::forward<VerifiedAccessEndpointIdsT>(value));
    return *this;
  }
  template <typename VerifiedAccessEndpointIdsT = Aws::String>
  DescribeVerifiedAccessEndpointsRequest& AddVerifiedAccessEndpointIds(VerifiedAccessEndpointIdsT&& value) {
    m_verifiedAccessEndpointIdsHasBeenSet = true;
    m_verifiedAccessEndpointIds.emplace_back(std::forward<VerifiedAccessEndpointIdsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of the Verified Access instance.</p>
   */
  inline const Aws::String& GetVerifiedAccessInstanceId() const { return m_verifiedAccessInstanceId; }
  inline bool VerifiedAccessInstanceIdHasBeenSet() const { return m_verifiedAccessInstanceIdHasBeenSet; }
  template <typename VerifiedAccessInstanceIdT = Aws::String>
  void SetVerifiedAccessInstanceId(VerifiedAccessInstanceIdT&& value) {
    m_verifiedAccessInstanceIdHasBeenSet = true;
    m_verifiedAccessInstanceId = std::forward<VerifiedAccessInstanceIdT>(value);
  }
  template <typename VerifiedAccessInstanceIdT = Aws::String>
  DescribeVerifiedAccessEndpointsRequest& WithVerifiedAccessInstanceId(VerifiedAccessInstanceIdT&& value) {
    SetVerifiedAccessInstanceId(std::forward<VerifiedAccessInstanceIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of the Verified Access group.</p>
   */
  inline const Aws::String& GetVerifiedAccessGroupId() const { return m_verifiedAccessGroupId; }
  inline bool VerifiedAccessGroupIdHasBeenSet() const { return m_verifiedAccessGroupIdHasBeenSet; }
  template <typename VerifiedAccessGroupIdT = Aws::String>
  void SetVerifiedAccessGroupId(VerifiedAccessGroupIdT&& value) {
    m_verifiedAccessGroupIdHasBeenSet = true;
    m_verifiedAccessGroupId = std::forward<VerifiedAccessGroupIdT>(value);
  }
  template <typename VerifiedAccessGroupIdT = Aws::String>
  DescribeVerifiedAccessEndpointsRequest& WithVerifiedAccessGroupId(VerifiedAccessGroupIdT&& value) {
    SetVerifiedAccessGroupId(std::forward<VerifiedAccessGroupIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of results to return with a single call. To retrieve the
   * remaining results, make another call with the returned <code>nextToken</code>
   * value.</p>
   */
  inline int GetMaxResults() const { return m_maxResults; }
  inline bool MaxResultsHasBeenSet() const { return m_maxResultsHasBeenSet; }
  inline void SetMaxResults(int value) {
    m_maxResultsHasBeenSet = true;
    m_maxResults = value;
  }
  inline DescribeVerifiedAccessEndpointsRequest& WithMaxResults(int value) {
    SetMaxResults(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The token for the next page of results.</p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  DescribeVerifiedAccessEndpointsRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>One or more filters. Filter names and values are case-sensitive.</p>
   */
  inline const Aws::Vector<Filter>& GetFilters() const { return m_filters; }
  inline bool FiltersHasBeenSet() const { return m_filtersHasBeenSet; }
  template <typename FiltersT = Aws::Vector<Filter>>
  void SetFilters(FiltersT&& value) {
    m_filtersHasBeenSet = true;
    m_filters = std::forward<FiltersT>(value);
  }
  template <typename FiltersT = Aws::Vector<Filter>>
  DescribeVerifiedAccessEndpointsRequest& WithFilters(FiltersT&& value) {
    SetFilters(std::forward<FiltersT>(value));
    return *this;
  }
  template <typename FiltersT = Filter>
  DescribeVerifiedAccessEndpointsRequest& AddFilters(FiltersT&& value) {
    m_filtersHasBeenSet = true;
    m_filters.emplace_back(std::forward<FiltersT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Checks whether you have the required permissions for the action, without
   * actually making the request, and provides an error response. If you have the
   * required permissions, the error response is <code>DryRunOperation</code>.
   * Otherwise, it is <code>UnauthorizedOperation</code>.</p>
   */
  inline bool GetDryRun() const { return m_dryRun; }
  inline bool DryRunHasBeenSet() const { return m_dryRunHasBeenSet; }
  inline void SetDryRun(bool value) {
    m_dryRunHasBeenSet = true;
    m_dryRun = value;
  }
  inline DescribeVerifiedAccessEndpointsRequest& WithDryRun(bool value) {
    SetDryRun(value);
    return *this;
  }
  ///@}
 private:
  Aws::Vector<Aws::String> m_verifiedAccessEndpointIds;
  bool m_verifiedAccessEndpointIdsHasBeenSet = false;

  Aws::String m_verifiedAccessInstanceId;
  bool m_verifiedAccessInstanceIdHasBeenSet = false;

  Aws::String m_verifiedAccessGroupId;
  bool m_verifiedAccessGroupIdHasBeenSet = false;

  int m_maxResults{0};
  bool m_maxResultsHasBeenSet = false;

  Aws::String m_nextToken;
  bool m_nextTokenHasBeenSet = false;

  Aws::Vector<Filter> m_filters;
  bool m_filtersHasBeenSet = false;

  bool m_dryRun{false};
  bool m_dryRunHasBeenSet = false;
};

}  // namespace Model
}  // namespace EC2
}  // namespace Aws
