﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/s3-crt/S3Crt_EXPORTS.h>
#include <aws/s3-crt/model/BucketLocationConstraint.h>
#include <aws/s3-crt/model/LocationInfo.h>
#include <aws/s3-crt/model/BucketInfo.h>
#include <utility>

namespace Aws
{
namespace Utils
{
namespace Xml
{
  class XmlNode;
} // namespace Xml
} // namespace Utils
namespace S3Crt
{
namespace Model
{

  /**
   * <p>The configuration information for the bucket.</p><p><h3>See Also:</h3>   <a
   * href="http://docs.aws.amazon.com/goto/WebAPI/s3-2006-03-01/CreateBucketConfiguration">AWS
   * API Reference</a></p>
   */
  class CreateBucketConfiguration
  {
  public:
    AWS_S3CRT_API CreateBucketConfiguration();
    AWS_S3CRT_API CreateBucketConfiguration(const Aws::Utils::Xml::XmlNode& xmlNode);
    AWS_S3CRT_API CreateBucketConfiguration& operator=(const Aws::Utils::Xml::XmlNode& xmlNode);

    AWS_S3CRT_API void AddToNode(Aws::Utils::Xml::XmlNode& parentNode) const;


    ///@{
    /**
     * <p>Specifies the Region where the bucket will be created. You might choose a
     * Region to optimize latency, minimize costs, or address regulatory requirements.
     * For example, if you reside in Europe, you will probably find it advantageous to
     * create buckets in the Europe (Ireland) Region.</p> <p>If you don't specify a
     * Region, the bucket is created in the US East (N. Virginia) Region (us-east-1) by
     * default. Configurations using the value <code>EU</code> will create a bucket in
     * <code>eu-west-1</code>.</p> <p>For a list of the valid values for all of the
     * Amazon Web Services Regions, see <a
     * href="https://docs.aws.amazon.com/general/latest/gr/rande.html#s3_region">Regions
     * and Endpoints</a>.</p>  <p>This functionality is not supported for
     * directory buckets.</p> 
     */
    inline const BucketLocationConstraint& GetLocationConstraint() const{ return m_locationConstraint; }
    inline bool LocationConstraintHasBeenSet() const { return m_locationConstraintHasBeenSet; }
    inline void SetLocationConstraint(const BucketLocationConstraint& value) { m_locationConstraintHasBeenSet = true; m_locationConstraint = value; }
    inline void SetLocationConstraint(BucketLocationConstraint&& value) { m_locationConstraintHasBeenSet = true; m_locationConstraint = std::move(value); }
    inline CreateBucketConfiguration& WithLocationConstraint(const BucketLocationConstraint& value) { SetLocationConstraint(value); return *this;}
    inline CreateBucketConfiguration& WithLocationConstraint(BucketLocationConstraint&& value) { SetLocationConstraint(std::move(value)); return *this;}
    ///@}

    ///@{
    /**
     * <p>Specifies the location where the bucket will be created.</p> <p> <b>Directory
     * buckets </b> - The location type is Availability Zone or Local Zone. To use the
     * Local Zone location type, your account must be enabled for Dedicated Local
     * Zones. Otherwise, you get an HTTP <code>403 Forbidden</code> error with the
     * error code <code>AccessDenied</code>. To learn more, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/opt-in-directory-bucket-lz.html">Enable
     * accounts for Dedicated Local Zones</a> in the <i>Amazon S3 User Guide</i>. </p>
     *  <p>This functionality is only supported by directory buckets.</p> 
     */
    inline const LocationInfo& GetLocation() const{ return m_location; }
    inline bool LocationHasBeenSet() const { return m_locationHasBeenSet; }
    inline void SetLocation(const LocationInfo& value) { m_locationHasBeenSet = true; m_location = value; }
    inline void SetLocation(LocationInfo&& value) { m_locationHasBeenSet = true; m_location = std::move(value); }
    inline CreateBucketConfiguration& WithLocation(const LocationInfo& value) { SetLocation(value); return *this;}
    inline CreateBucketConfiguration& WithLocation(LocationInfo&& value) { SetLocation(std::move(value)); return *this;}
    ///@}

    ///@{
    /**
     * <p>Specifies the information about the bucket that will be created.</p> 
     * <p>This functionality is only supported by directory buckets.</p> 
     */
    inline const BucketInfo& GetBucket() const{ return m_bucket; }
    inline bool BucketHasBeenSet() const { return m_bucketHasBeenSet; }
    inline void SetBucket(const BucketInfo& value) { m_bucketHasBeenSet = true; m_bucket = value; }
    inline void SetBucket(BucketInfo&& value) { m_bucketHasBeenSet = true; m_bucket = std::move(value); }
    inline CreateBucketConfiguration& WithBucket(const BucketInfo& value) { SetBucket(value); return *this;}
    inline CreateBucketConfiguration& WithBucket(BucketInfo&& value) { SetBucket(std::move(value)); return *this;}
    ///@}
  private:

    BucketLocationConstraint m_locationConstraint;
    bool m_locationConstraintHasBeenSet = false;

    LocationInfo m_location;
    bool m_locationHasBeenSet = false;

    BucketInfo m_bucket;
    bool m_bucketHasBeenSet = false;
  };

} // namespace Model
} // namespace S3Crt
} // namespace Aws
