/*
 * SPDX-FileCopyrightText: Copyright (c) 2021-2025 NVIDIA CORPORATION & AFFILIATES. All rights reserved.
 * SPDX-License-Identifier: Apache-2.0 WITH LLVM-exception
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * Licensed under the Apache License v2.0 with LLVM Exceptions.
 * See https://nvidia.github.io/NVTX/LICENSE.txt for license information.
 */

#ifndef NVTX_EXT_IMPL_COLLECTOR_GUARD
#error Never include this file directly -- it is automatically included by nvToolsExtCollector.h (except when NVTX_NO_IMPL is defined).
#endif

#define NVTX_EXT_IMPL_GUARD
#include "nvtxExtImpl.h"
#undef NVTX_EXT_IMPL_GUARD

#ifndef NVTX_EXT_IMPL_COLLECTOR_V1
#define NVTX_EXT_IMPL_COLLECTOR_V1

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

#ifdef NVTX_DISABLE

#include "nvtxExtHelperMacros.h"

#define NVTX_EXT_COLLECTOR_IMPL_FN_V1(ret_type, fn_name, signature, arg_names) \
ret_type fn_name signature { \
    NVTX_SET_NAME_MANGLING_OPTIONS \
    NVTX_EXT_HELPER_UNUSED_ARGS arg_names \
    NVTX_EXT_FN_RETURN_INVALID(ret_type) \
}

#else /* NVTX_DISABLE */

/*
 * Function slots for the collector extension. First entry is the module state,
 * initialized to `0` (`NVTX_EXTENSION_FRESH`).
 */
#define NVTX_EXT_COLLECTOR_SLOT_COUNT 63

NVTX_LINKONCE_DEFINE_GLOBAL intptr_t
NVTX_EXT_COLLECTOR_VERSIONED_ID(nvtxExtCollectorSlots)[NVTX_EXT_COLLECTOR_SLOT_COUNT + 1]
    = {0};

/* Avoid warnings about missing prototype. */
NVTX_LINKONCE_FWDDECL_FUNCTION void NVTX_EXT_COLLECTOR_VERSIONED_ID(nvtxExtCollectorInitOnce)(void);
NVTX_LINKONCE_DEFINE_FUNCTION void NVTX_EXT_COLLECTOR_VERSIONED_ID(nvtxExtCollectorInitOnce)(void)
{
    intptr_t* fnSlots = NVTX_EXT_COLLECTOR_VERSIONED_ID(nvtxExtCollectorSlots) + 1;
    nvtxExtModuleSegment_t segment = {
        0, /* unused (only one segment) */
        NVTX_EXT_COLLECTOR_SLOT_COUNT,
        fnSlots
    };

    nvtxExtModuleInfo_t module = {
        NVTX_VERSION, sizeof(nvtxExtModuleInfo_t),
        NVTX_EXT_COLLECTOR_MODULEID, NVTX_EXT_COLLECTOR_COMPATID,
        1, &segment, /* number of segments, segments */
        NULL, /* no export function needed */
        NULL /* no extension private info */
    };

    NVTX_INFO( "%s\n", __FUNCTION__  );

    NVTX_VERSIONED_IDENTIFIER(nvtxExtInitOnce)(&module,
        NVTX_EXT_COLLECTOR_VERSIONED_ID(nvtxExtCollectorSlots));
}

#define NVTX_EXT_COLLECTOR_IMPL_FN_V1(ret_type, fn_name, signature, arg_names) \
typedef ret_type (*fn_name##_impl_fntype)signature; \
NVTX_DECLSPEC ret_type NVTX_API fn_name signature { \
    NVTX_SET_NAME_MANGLING_OPTIONS \
    intptr_t* pSlot = &NVTX_EXT_COLLECTOR_VERSIONED_ID(nvtxExtCollectorSlots)[NVTX3EXT_CBID_##fn_name + 1]; \
    intptr_t slot = *pSlot; \
    if (slot != NVTX_EXTENSION_DISABLED) { \
        if (slot != NVTX_EXTENSION_FRESH) { \
            NVTX_EXT_FN_RETURN (*(fn_name##_impl_fntype)slot) arg_names; \
        } else { \
            NVTX_EXT_COLLECTOR_VERSIONED_ID(nvtxExtCollectorInitOnce)(); \
            /* Re-read function slot after extension initialization. */ \
            slot = *pSlot; \
            if (slot != NVTX_EXTENSION_DISABLED && slot != NVTX_EXTENSION_FRESH) { \
                NVTX_EXT_FN_RETURN (*(fn_name##_impl_fntype)slot) arg_names; \
            } \
        } \
    } \
    NVTX_EXT_FN_RETURN_INVALID(ret_type) /* No tool attached. */ \
}

#endif /* NVTX_DISABLE */

/* Non-void functions. */
#define NVTX_EXT_FN_RETURN return
#define NVTX_EXT_FN_RETURN_INVALID(rtype) return ((rtype)NVTX_COLLECTOR_STATE_INVALID);

NVTX_EXT_COLLECTOR_IMPL_FN_V1(nvtxCollectorState_t, nvtxCollectorGetState,
    (void),
    ())

#undef NVTX_EXT_FN_RETURN
#undef NVTX_EXT_FN_RETURN_INVALID

#define NVTX_EXT_FN_RETURN return
#define NVTX_EXT_FN_RETURN_INVALID(rtype) return ((rtype)0);

NVTX_EXT_COLLECTOR_IMPL_FN_V1(nvtxCollectorFinalizationHandle, nvtxCollectorAcquireFinalizationToken,
    (void),
    ())

#undef NVTX_EXT_FN_RETURN
#undef NVTX_EXT_FN_RETURN_INVALID

#define NVTX_EXT_FN_RETURN return
#define NVTX_EXT_FN_RETURN_INVALID(rtype) return ((rtype)NVTX_FAIL);

NVTX_EXT_COLLECTOR_IMPL_FN_V1(int, nvtxCollectorReleaseFinalizationToken,
    (nvtxCollectorFinalizationHandle token),
    (token))

NVTX_EXT_COLLECTOR_IMPL_FN_V1(int, nvtxCollectorWaitForState,
    (nvtxCollectorState_t state, uint64_t timeoutNsec),
    (state, timeoutNsec))

#undef NVTX_EXT_FN_RETURN
#undef NVTX_EXT_FN_RETURN_INVALID
/* END: Non-void functions. */

/* Keep NVTX_EXT_COLLECTOR_IMPL_FN_V1 defined for a future version of this extension. */

#ifdef __cplusplus
} /* extern "C" */
#endif /* __cplusplus */

#endif /* NVTX_EXT_IMPL_COLLECTOR_V1 */
