/**
 * @license
 * Copyright 2025 Google LLC
 * SPDX-License-Identifier: Apache-2.0
 */
let detectionComplete = false;
let protocolSupported = false;
let protocolEnabled = false;
let sgrMouseEnabled = false;
/**
 * Detects Kitty keyboard protocol support.
 * Definitive document about this protocol lives at https://sw.kovidgoyal.net/kitty/keyboard-protocol/
 * This function should be called once at app startup.
 */
export async function detectAndEnableKittyProtocol() {
    if (detectionComplete) {
        return protocolSupported;
    }
    return new Promise((resolve) => {
        if (!process.stdin.isTTY || !process.stdout.isTTY) {
            detectionComplete = true;
            resolve(false);
            return;
        }
        const originalRawMode = process.stdin.isRaw;
        if (!originalRawMode) {
            process.stdin.setRawMode(true);
        }
        let responseBuffer = '';
        let progressiveEnhancementReceived = false;
        let timeoutId;
        const onTimeout = () => {
            timeoutId = undefined;
            process.stdin.removeListener('data', handleData);
            if (!originalRawMode) {
                process.stdin.setRawMode(false);
            }
            detectionComplete = true;
            resolve(false);
        };
        const handleData = (data) => {
            if (timeoutId === undefined) {
                // Race condition. We have already timed out.
                return;
            }
            responseBuffer += data.toString();
            // Check for progressive enhancement response (CSI ? <flags> u)
            if (responseBuffer.includes('\x1b[?') && responseBuffer.includes('u')) {
                progressiveEnhancementReceived = true;
                // Give more time to get the full set of kitty responses if we have an
                // indication the terminal probably supports kitty and we just need to
                // wait a bit longer for a response.
                clearTimeout(timeoutId);
                timeoutId = setTimeout(onTimeout, 1000);
            }
            // Check for device attributes response (CSI ? <attrs> c)
            if (responseBuffer.includes('\x1b[?') && responseBuffer.includes('c')) {
                clearTimeout(timeoutId);
                timeoutId = undefined;
                process.stdin.removeListener('data', handleData);
                if (!originalRawMode) {
                    process.stdin.setRawMode(false);
                }
                if (progressiveEnhancementReceived) {
                    // Enable the protocol
                    process.stdout.write('\x1b[>1u');
                    protocolSupported = true;
                    protocolEnabled = true;
                }
                // Broaden mouse support by enabling SGR mode if we get any device
                // attribute response, which is a strong signal of a modern terminal.
                process.stdout.write('\x1b[?1006h');
                sgrMouseEnabled = true;
                // Set up cleanup on exit for all enabled protocols
                process.on('exit', disableAllProtocols);
                process.on('SIGTERM', disableAllProtocols);
                detectionComplete = true;
                resolve(protocolSupported);
            }
        };
        process.stdin.on('data', handleData);
        // Send queries
        process.stdout.write('\x1b[?u'); // Query progressive enhancement
        process.stdout.write('\x1b[c'); // Query device attributes
        // Timeout after 200ms
        // When a iterm2 terminal does not have focus this can take over 90s on a
        // fast macbook so we need a somewhat longer threshold than would be ideal.
        timeoutId = setTimeout(onTimeout, 200);
    });
}
function disableAllProtocols() {
    if (protocolEnabled) {
        process.stdout.write('\x1b[<u');
        protocolEnabled = false;
    }
    if (sgrMouseEnabled) {
        process.stdout.write('\x1b[?1006l'); // Disable SGR Mouse
        sgrMouseEnabled = false;
    }
}
export function isKittyProtocolEnabled() {
    return protocolEnabled;
}
export function isKittyProtocolSupported() {
    return protocolSupported;
}
//# sourceMappingURL=kittyProtocolDetector.js.map