# Copyright (C) 2024 Intel Corporation
# SPDX-License-Identifier: MIT
from pathlib import Path
from typing import Union, List

from mpp.console_output import ConsoleOutput
from mpp.core.data_processor import _DataProcessor, DataProcessorFactory
from mpp import ViewCollection, ViewInitializer, ViewGenerator, DataAccumulator, Partition
from mpp.core.devices import DeviceCollection
from mpp.core.api_args import ApiArgs, SystemInformation


class MppApi:

    def __init__(self, args: ApiArgs):
        self.__args = args
        self.__device_collection: DeviceCollection = None
        self.__view_collection: ViewCollection = None
        self.__view_generator: ViewGenerator = None
        self.data_accumulator: DataAccumulator = None
        self.__data_processor: _DataProcessor = None
        self.__view_writer: 'ViewWriter' = None
        self.__total_metrics_derived = 0
        self.__summary_views = None

        self.__system_information: SystemInformation = args.system_information

    @property
    def device_collection(self) -> DeviceCollection:
        return self.__device_collection

    @property
    def view_collection(self) -> ViewCollection:
        return self.__view_collection

    @property
    def view_generator(self) -> ViewGenerator:
        return self.__view_generator

    @property
    def view_writer(self) -> 'ViewWriter':
        return self.__view_writer
    
    @property
    def total_metrics_derived(self):
        return self.__total_metrics_derived
    
    @property
    def summary_views(self):
        return self.__summary_views

    @property
    def ref_tsc(self):
        return self.__system_information.ref_tsc

    @property
    def detail_views(self):
        return self.__data_processor.detail_views

    def initialize(self):
        self.__initialize_metric_computer_map()
        self.__device_collection = DeviceCollection(self.__args)
        self.__device_collection.initialize_devices()
        self.__initialize_view_collection()
        self.__view_generator = ViewGenerator(self.view_collection)
        self.data_accumulator = DataAccumulator(self.__view_generator)
        self.__view_writer = ViewWriterFactory().create(self.__system_information.has_modules,
                                                        self.__args.output_writers)
        self.__data_processor = DataProcessorFactory().create(self.__args.is_parallel, self.data_accumulator,
                                                              self.view_generator, self.view_writer,
                                                              self.__args.event_reader, self.__args.verbose)

    def process_partitions(self,
                           partitions: List['Partition'],
                           number_of_cores=1,
                           generate_detail_views=True):
        self.__data_processor.process_partitions(partitions, number_of_cores, generate_detail_views)

    def process_partition(self, partition: 'Partition'):
        self.__data_processor.process_partition(partition)

    def generate_summary_views(self, first_sample, last_sample, timer=None):
        if timer:
            with timer() as number_of_seconds:
                self.__generate_summary_views(first_sample, last_sample)
                if ConsoleOutput.is_regular_verbosity(self.__args.verbose):
                    print(f'Generated all summary tables in {number_of_seconds}')
        else:
            self.__generate_summary_views(first_sample, last_sample)

    def __initialize_metric_computer_map(self):
        self.__args.metric_computer_map = _MetricComputerMapGenerator(self.__args).generate()

    def __generate_summary_views(self, first_sample, last_sample):
        self.__summary_views = self.data_accumulator.generate_summary_views()
        self.__total_metrics_derived = self._get_total_derived_metrics()
        if self.view_writer:
            self.view_writer.write(list(self.__summary_views.values()), first_sample, last_sample)

    def _get_total_derived_metrics(self):
        view_names = list(filter(lambda x: 'system_view_summary' in x, list(self.__summary_views.keys())))
        metric_names = set()
        for view in view_names:
            metric_names = metric_names.union(set(list(filter(lambda x: 'metric_' in x, self.__summary_views[view].data.columns))))
        return metric_names

    def __initialize_view_collection(self):
        view_initializer = ViewInitializer(self.__device_collection.devices, self.__args)
        view_initializer.add_views()
        self.__view_collection = view_initializer.view_collection


class _MetricComputerMapGenerator:

    def __init__(self, args: ApiArgs):
        self.__args = args
        self.__metric_computer_map = {}

    def generate(self):
        for core_type, metric_file in self.__args.metric_file_map.items():
            self.__metric_computer_map[core_type] = self.__get_metric_computer(core_type, metric_file)
        return self.__metric_computer_map

    def __get_metric_computer(self, core_type, metric_file):
        from mpp.parsers.collector_detection import SymbolTableFactory
        from mpp.parsers.metrics import MetricDefinitionParserFactory
        from mpp.core.metric_computer import MetricComputer
        symbol_table = SymbolTableFactory.create(self.__args.collector, self.__args.system_information, core_type,
                                                 self.__args.retire_latency)
        metric_definitions = MetricDefinitionParserFactory.create(Path(metric_file)).parse()
        metric_computer = MetricComputer(metric_definitions, symbol_table)
        return metric_computer


class NullViewWriter:
    """
    A ViewWriter that does nothing
    """
    # TODO: move ViewWriter into mpp package

    def write(self, views, first_sample: int, last_sample: int) -> None:
        pass

    def remove(self, views) -> None:
        pass


class ViewWriterFactory:
    """
    Factory class for creating ViewWriter objects
    """

    @staticmethod
    def create(show_modules: bool, writers: Union['OutputWriter', List['OutputWriter']]) -> (
            'ViewWriter'):
        """
        Create a new ViewWriter object

        :param show_modules: whether to show module information in the output
        :param writers: a single writer object, or a list of writer objects, to use for outputting data

        :return: a new ViewWriter object
        """
        if writers:
            from cli.writers.views import ViewWriter
            return ViewWriter(show_modules, writers)
        else:
            return NullViewWriter()
