# Copyright (C) 2024 Intel Corporation
# SPDX-License-Identifier: MIT

from abc import ABC, abstractmethod
from pathlib import Path


class ArchiveExtractor(ABC):
    @staticmethod
    @abstractmethod
    def extract(src, dst):
        pass


class WindowsArchiveExtractor(ArchiveExtractor):
    @staticmethod
    def extract(src, dst):
        import zipfile
        with zipfile.ZipFile(src, 'r') as archive:
            archive.extractall(dst)


class LinuxArchiveExtractor(ArchiveExtractor):
    @staticmethod
    def extract(src, dst):
        import tarfile
        with tarfile.open(src) as archive:
            archive.extractall(dst)


class ArchiveExtractorFactory:

    archive_extractor = {
        '.zip': WindowsArchiveExtractor,
        '.tar.gz': LinuxArchiveExtractor,
        '.gz': LinuxArchiveExtractor
    }

    @classmethod
    def create(cls, archive_path: Path):
        if archive_path.suffix not in cls.archive_extractor:
            raise ValueError(f'No archiver found for {archive_path.suffix}')
        return cls.archive_extractor[archive_path.suffix]()